import {
  OnInit,
  AfterContentInit,
  Component,
  Input,
  ViewChild
} from '@angular/core';
import { OgmaService } from './ogma.service';

import { NodeId, InputTarget } from 'ogma';
import initialGraph from './data';

interface HoverEvent {
  target: InputTarget;
  x: number;
  y: number;
};

@Component({
  selector: 'app-root',
  templateUrl: './app.component.html',
  styleUrls: ['./app.component.css']
})
export class AppComponent implements OnInit, AfterContentInit {
  @ViewChild('ogmaContainer', { static: true })
  private container;

  @Input() currentLayout: string = 'force';
  layouts: string[] = ['force', 'hierarchical'];

  hoveredContent: { id: NodeId; type: string };
  hoveredPosition: { x: number; y: number };

  /**
   * @param {OgmaService} ogmaService
   */
  constructor(private ogmaService: OgmaService) { }

  /**
   * Initialize the configuration of Ogma
   */
  ngOnInit() {
    this.ogmaService.initConfig({
      graph: initialGraph,
      options: {
        backgroundColor: 'rgb(240, 240, 240)'
      }
    });

    this.ogmaService.ogma.events.onHover(({ x, y, target }: HoverEvent) => {
      if (target.isNode) {
        // save the tooltip state (offset by 20px the vertical position)
        this.hoveredContent = {
          id: target.getId(),
          type: target.getAttribute('color')
        };
        this.hoveredPosition = { x, y: y + 20 };
      }
    });

    this.ogmaService.ogma.events.onUnhover((_: HoverEvent) => {
      // clear the tooltip state
      this.hoveredContent = null;
    });
  }

  /**
   * Ogma container must be set when content is initialized
   */
  ngAfterContentInit() {
    this.ogmaService.ogma.setContainer(this.container.nativeElement);
    return this.runLayout();
  }

  /**
   * Add a new node to the visualization
   */
  public addNode(): Promise<void> {
    this.ogmaService.addNewNode();
    return this.runLayout();
  }

  /**
   * Runs a layout with the current graph
   */
  public runLayout(): Promise<void> {
    return this.ogmaService.runLayout(this.currentLayout);
  }

  /**
   * Randomize geo coordinates & toggle geo mode
   */
  public toggleGeo() {
    return this.ogmaService.toggleGeoMode();
  }

  /**
   * Returns the number of nodes in the graph
   */
  public countNodes(): number {
    return this.ogmaService.ogma.getNodes().size;
  }
}
