import { Injectable } from '@angular/core';
import Ogma, { RawGraph, RendererType, Options, CrossOriginValue, RawNode, RawEdge } from 'ogma';
import * as L from 'leaflet';

interface OgmaParameters {
  dimensions?: {
    width?: number;
    height?: number;
  };
  renderer?: RendererType;
  options?: Options;
  container?: HTMLElement | string;
  graph?: RawGraph;
  imgCrossOrigin?: CrossOriginValue;
}

Ogma.libraries['leaflet'] = L;

function createNode(id: number): RawNode {
  return {
    id,
    attributes: {
      color: id % 2 ? 'purple' : 'orange'
    }
  };
}

function createEdge(node: RawNode, ids: RawNode['id'][]): RawEdge {
  // pick a random node in the graph
  const randomIndex = Math.floor(Math.random() * ids.length);
  const otherNode = ids[randomIndex];
  return { id: `${otherNode}-${node.id}`, source: otherNode, target: node.id };
}

@Injectable()
export class OgmaService {
  // expose an instance of Ogma from the service
  public ogma: Ogma;


  public initConfig(configuration: OgmaParameters = {}) {
    this.ogma = new Ogma(configuration);
  }

  public addNewNode() {
    const idsInGraph = this.ogma.getNodes().getId();
    const newNode = createNode(idsInGraph.length);

    this.ogma.addGraph({
      nodes: [newNode],
      edges: [createEdge(newNode, idsInGraph)]
    });
  }

  private _randomizeCoordinates() {
    const nodes = this.ogma.getNodes();
    nodes.setGeoCoordinates(
      nodes.map(() => ({
        latitude: 30 + Math.random() * 10,
        longitude: 0 + Math.random() * 15
      }))
    )
  }

  public runLayout(layout: string = 'force'): Promise<void> {
    return this.ogma.layouts[layout]({ locate: true });
  }

  public toggleGeoMode() {
    this._randomizeCoordinates();
    return this.ogma.geo.enabled() ?
      this.ogma.geo.disable() :
      this.ogma.geo.enable();
  }
}
