---
name: ogma-skill
description: Guide to building interactive graph visualization applications using the Ogma library.
license: Apache-2.0
metadata:
  author: linkurious-ogma
  version: "5.3.11"
---

# Ogma Graph Visualization

Build interactive graph visualization applications using the Ogma library.

## Overview

Ogma is a JavaScript/TypeScript library for visualizing and interacting with graphs (networks of nodes and edges). It provides WebGL/Canvas/SVG rendering, layout algorithms, styling, transformations, and geographic visualization.

## When to Use This Skill

- Creating new graph visualization applications
- Adding interactive network diagrams to web apps
- Improving existing Ogma-based applications
- Implementing graph layouts, styling, or transformations
- Building React or Vue applications with graph visualization

## Key Resources

- [API Reference](references/REFERENCE.md) - Complete API documentation
- [Core API](concepts/core-api.md) - Graph, nodes, edges, data management
- [Styles](concepts/styles.md) - Visual styling and rules
- [Layouts](concepts/layouts.md) - Force, hierarchical, radial layouts
- [Events](concepts/events.md) - Interactions and event handling
- [Transformations](concepts/transformations.md) - Grouping and filtering
- [Tools](concepts/tools.md) - Additional utilities and features

## Quick Start

### Basic Setup (TypeScript)

```typescript
import Ogma from "@linkurious/ogma";

// Create Ogma instance
const ogma = new Ogma({
  container: "graph-container", // DOM element ID or HTMLElement
});

// Add graph data
await ogma.setGraph({
  nodes: [
    { id: "n1", data: { label: "Node 1" } },
    { id: "n2", data: { label: "Node 2" } },
  ],
  edges: [{ id: "e1", source: "n1", target: "n2" }],
});

// Apply layout and center view
await ogma.layouts.force();
await ogma.view.locateGraph();
```

### React Setup

```tsx
import { Ogma, NodeStyle } from "@linkurious/ogma-react";
import type { RawGraph } from "@linkurious/ogma";

function App() {
  const graph: RawGraph = {
    nodes: [{ id: "n1" }, { id: "n2" }],
    edges: [{ id: "e1", source: "n1", target: "n2" }],
  };

  return (
    <Ogma graph={graph} onReady={() => console.log("Ready!")}>
      <NodeStyle attributes={{ color: "#61dafb", radius: 10 }} />
    </Ogma>
  );
}
```

### Vue Setup

```vue
<template>
  <div id="graph-container" style="width: 100%; height: 100%;"></div>
</template>

<script setup lang="ts">
import Ogma from "@linkurious/ogma";
import { ref, onMounted, onUnmounted } from "vue";
import type { RawGraph } from "@linkurious/ogma";

const props = defineProps<{ data: RawGraph }>();
const ogma = ref<Ogma | null>(null);

onMounted(async () => {
  ogma.value = new Ogma({ container: "graph-container" });
  await ogma.value.setGraph(props.data);
  await ogma.value.layouts.force({ locate: true });
});

onUnmounted(() => ogma.value?.destroy());
</script>
```

## Common Workflows

### 1. Load and Display Graph Data

```typescript
// From JSON file
const graph = await Ogma.parse.jsonFromUrl("data/graph.json");
await ogma.setGraph(graph);

// From raw data
await ogma.setGraph({
  nodes: [{ id: 1 }, { id: 2 }],
  edges: [{ source: 1, target: 2 }],
});

// Run layout and center view on graph
await ogma.layouts.force({ locate: true });
```

### 2. Apply Data-Driven Styles

```typescript
// Style nodes based on data
ogma.styles.addNodeRule({
  color: (node) => (node.getData("type") === "important" ? "red" : "blue"),
  radius: (node) => 5 + node.getDegree(),
  text: { content: (node) => node.getData("name") },
});

// Style edges
ogma.styles.addEdgeRule({
  color: (edge) => (edge.getData("weight") > 10 ? "green" : "gray"),
  width: (edge) => Math.log(edge.getData("weight") + 1),
});
```

### 3. Apply Layout

```typescript
// Force-directed layout (good for most graphs)
await ogma.layouts.force();

// Hierarchical layout (for trees/DAGs)
await ogma.layouts.hierarchical({ direction: "TB" });

// Radial layout (centered on specific node)
await ogma.layouts.radial({ centerNode: ogma.getNode("root") });
```

### 4. Handle User Interactions

```typescript
// Click events
ogma.events.on("click", ({ target }) => {
  if (target?.isNode) {
    console.log("Clicked node:", target.getId(), target.getData());
  }
});

// Selection
ogma.events.on("nodesSelected", ({ nodes }) => {
  console.log("Selected:", nodes.getId());
});

// Hover
ogma.events.on("mouseover", ({ target }) => {
  if (target?.isNode) {
    target.setAttributes({ outerStroke: { color: "yellow", width: 3 } });
  }
});
```

### 5. Group Nodes

```typescript
// Group nodes by a property
const grouping = ogma.transformations.addNodeGrouping({
  groupIdFunction: (node) => node.getData("category"),
  nodeGenerator: (nodes, groupId) => ({
    id: `group-${groupId}`,
    data: { label: groupId, count: nodes.size },
    attributes: {
      radius: 10 + nodes.size * 2,
      text: { content: `${groupId} (${nodes.size})` },
    },
  }),
});

// Toggle grouping
await grouping.toggle();
```

## Templates

See the [templates](assets/) folder for complete working examples:

- Basic setup and data loading
- Node and edge styling
- Layout algorithms
- Event handling
- Grouping transformations
- React/Vue integrations

## Type Safety

Ogma supports generic types for node and edge data:

```typescript
interface NodeData {
  name: string;
  category: "person" | "company";
  revenue?: number;
}

interface EdgeData {
  relationship: string;
  weight: number;
}

const ogma = new Ogma<NodeData, EdgeData>({
  container: "graph-container",
});

// Now getData() is typed
const node = ogma.getNode("n1");
const name: string = node.getData("name"); // TypeScript knows this is string
```

**Use modern features** prefer `async/await` over promises for better readability and performance and pay attention to the API return types, most of Ogma methods return `Promise`s

## Performance Tips

1. **Use WebGL renderer** (default) for graphs > 2,000 nodes
2. **Batch operations** when adding many nodes/edges
3. **Use `duration: 0`** to skip animations for large graphs
4. **Apply transformations** (filtering, grouping) to reduce visible elements
5. **Use GPU-accelerated force layout** for very large graphs:
   ```typescript
   await ogma.layouts.force({ gpu: true });
   ```
6. **Make sure style rules are efficient** - avoid complex functions in style rules

## Framework Integration

- **React**: Use `@linkurious/ogma-react` - see [React wrapper tutorial](https://doc.linkurious.com/ogma/latest/tutorials/react-wrapper/)
- **Vue**: Use `@linkurious/ogma-vue` - see [Vue tutorial](https://doc.linkurious.com/ogma/latest/tutorials/vue/)
- **Angular**: See [Angular tutorial](https://doc.linkurious.com/ogma/latest/tutorials/angular/)
- **Typescript**: Usde Fully typed API for type-safe development, prefer `async/await` syntax and type the Node and Edge data interfaces.
