# Data-driven styles

This example shows how to use nodes and edges [data](/api/node.html#node-getdata) as well as rule [slice](/api/ogma/rules.html#ogma-rules-slices) and [map](/api/ogma/rules.html#ogma-rules-map) to style your graph.<br/>

## Code

```typescript
import Ogma from '@linkurious/ogma';
import { getIconCode } from './utils';

const fontFamily = 'IBM Plex Sans';

interface NodeData {
  properties: {
    name: string;
    funding_total?: number;
    picture: string;
  };
  categories: ('INVESTOR' | 'MARKET' | 'COMPANY' | 'PERSON')[];
}

interface EdgeData {
  properties: {
    raised_amount_usd: number;
  };
  type: 'INVESTED_IN' | 'ACQUIRED' | 'HAS_MARKET';
}

const ogma = new Ogma<NodeData, EdgeData>({
  container: 'graph-container'
});

// Load data from a json file.
const graph = await Ogma.parse.jsonFromUrl<NodeData, EdgeData>(
  'solarCity-nostyle.json'
);
await ogma.setGraph(graph);
await ogma.view.locateGraph();

// Define the Node style rules
ogma.styles.addNodeRule({
  // the label is the value os the property name.
  text: {
    content: node => node.getData('properties.name'),
    font: fontFamily,
    size: 14
  },

  // the size is proportional to the funding_total property
  radius: ogma.rules.slices({
    field: 'properties.funding_total',
    values: { nbSlices: 7, min: 3, max: 10 }
  }),

  // the color is based on the funding_total property (from red to purple)
  color: ogma.rules.slices({
    field: 'properties.funding_total',
    values: [
      '#161344',
      '#453455',
      '#632654',
      '#86315b',
      '#a93c63',
      '#cd476a',
      '#E94E5E'
    ],
    fallback: '#F79970'
  }),

  // assign icons based on the node category
  icon: {
    content: ogma.rules.map({
      field: 'categories',
      values: {
        COMPANY: getIconCode('icon-rocket'),
        INVESTOR: getIconCode('icon-landmark'),
        MARKET: getIconCode('icon-gem'),
        PERSON: getIconCode('icon-user-round')
      }
    }),
    font: 'Lucide',
    scale: 0.6,
    style: 'bold',
    color: 'white'
  }
});

// Define the Edge style rules
ogma.styles.addEdgeRule({
  // the label is the value os the property name.
  text: {
    content: edge => edge.getData('type'),
    font: fontFamily
  },

  // the size is proportional to the raised_amount_usd property
  width: ogma.rules.slices({
    field: 'properties.raised_amount_usd',
    values: { nbSlices: 3, min: 0.75, max: 3 }
  }),

  // the  color is based on the raised_amount_usd property (from blue to black)
  color: ogma.rules.slices({
    field: 'properties.raised_amount_usd',
    values: ['#EB7D7D', '#E94E5E', '#132b43'],
    fallback: '#38040E',
    reverse: true
  }),

  // the shape is based on the type of edge
  shape: ogma.rules.map({
    field: 'type',
    values: {
      INVESTED_IN: 'arrow',
      ACQUIRED: 'tapered',
      HAS_MARKET: 'dotted'
    }
  })
});
```

## See Also

- [Live Example](https://doc.linkurio.us/ogma/latest/examples/data-driven-styles.html)
- [API Reference](references/REFERENCE.md)
