# Group edges

This example shows how to group parrallel edges by using the [edgeGrouping](/api/ogma/transformations.html#ogma-transformations-addedgegrouping) transformation.

## Code

```typescript
import Ogma, { Color, EdgeGrouping } from '@linkurious/ogma';
import { random } from './randomize';

const NB_EDGES = 10;
const COUNTRY = ['France', 'Russia'] as const;
type Country = (typeof COUNTRY)[number];

type NodeData = any;
type EdgeData = { country: Country };

const ogma = new Ogma<NodeData, EdgeData>({
  container: 'graph-container'
});

const COLORS: Record<Country, Color> = {
  France: '#de1d4d',
  Russia: '#1d60de'
};

ogma.addNodes([
  { id: 'n0', attributes: { x: -50, y: -50 } },
  { id: 'n1', attributes: { x: 50, y: 50 } }
]);

ogma.styles.addEdgeRule({
  color: edge => COLORS[edge.getData('country')] || 'grey'
});

ogma.view.locateGraph();

for (let i = 0; i < NB_EDGES; i++) {
  const country = COUNTRY[(random() * COUNTRY.length) | 0];
  ogma.addEdge({
    id: 'e' + i,
    source: 'n0',
    target: 'n1',
    attributes: {
      text: country,
      width: 2
    },
    data: { country }
  });
}

const input = document.getElementById('animation') as HTMLInputElement;
const button = document.getElementById('group-btn') as HTMLButtonElement;


// Groups all parallel edges that have the same data.country property
// into meta-edges which the size is the sum of every edge and
// the text is the content of data.country:
const transformation: EdgeGrouping<EdgeData, NodeData> = ogma.transformations.addEdgeGrouping({
  groupIdFunction: edge => edge.getData('country'),
  generator: (edges, groupId) => ({
    data: {
      subEdges: edges.getId(),
      country: edges.getData('country')[0]
    },
    attributes: {
      width: edges.reduce((width, edge) => {
        return Number(edge.getAttribute('width')) + width;
      }, 0),
      text: groupId
    }
  }),
  enabled: false
});

button.addEventListener('click', () => {
  const duration = input.checked ? 500 : 0;
  // Disable the button to prevent multiple clicks during the transformation
  button.disabled = true;
  transformation.toggle(duration).then(() => {
    const buttonText = transformation?.isEnabled()
      ? 'Ungroup edges'
      : 'Group edges';
    button.textContent = buttonText;

    // Re-enable the button after the transformation is done
    button.disabled = false;
  });
});

input.addEventListener('change', () => {
  const duration = input.checked ? 500 : 0;
  transformation.getAnimation().duration = duration;
});

ogma.events.on('click', evt => {
  if (evt.target && !evt.target.isNode) {
    const subEdges = evt.target.getData('subEdges');
    if (subEdges) console.log('sub edges:', subEdges);
  }
});
```

## Tags

transformations, grouping

## See Also

- [Live Example](https://doc.linkurio.us/ogma/latest/examples/group-edges.html)
- [API Reference](references/REFERENCE.md)
