# Group nodes

This example shows how to use the [groupNodes](/api/ogma/transformations.html#ogma-transformations-addnodegrouping) transformation to group nodes depending on a criteria. In this case, we will group nodes by their `location` data property.

## Code

```javascript
import Ogma from '@linkurious/ogma';
import { random } from './randomize';

const LOCATION = ['France', 'Russia', 'USA'];

const FLAGS = {
  France: 'flags/fr.svg',
  Russia: 'flags/ru.svg',
  USA: 'flags/us.svg'
};

const duration = 300;

const randomGraph = (N, E) => {
  const g = { nodes: [], edges: [] };

  for (let i = 0; i < N; i++) {
    const location = LOCATION[i % LOCATION.length];
    g.nodes.push({
      id: 'n' + i,
      attributes: {
        x: random() * 100,
        y: random() * 100,
        text: 'Node ' + i + ' - ' + location,
        image: {
          url: FLAGS[location]
        }
      },
      data: {
        location
      }
    });
  }

  for (let i = 0; i < E; i++) {
    g.edges.push({
      id: 'e' + i,
      source: 'n' + ((random() * N) | 0),
      target: 'n' + ((random() * N) | 0)
    });
  }

  return g;
};

const g = randomGraph(10, 10);

const ogma = new Ogma({
  graph: g,
  container: 'graph-container'
});

ogma.styles.addNodeRule({
  innerStroke: {
    color: '#999'
  },
  badges: {
    bottomRight: {
      stroke: {
        color: '#999'
      }
    }
  }
});

// Groups all active nodes that have the same `location` data property
// into a single node that will have the combined size of all nodes,
// and be at the center of the grouped nodes.
const transformation = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => node.getData('location'),
  nodeGenerator: (nodes, groupId) => {
    return {
      id: 'special group ' + groupId,
      data: {
        groupId: groupId,
        subNodes: nodes
      },
      attributes: {
        radius: nodes.reduce((acc, node) => {
          return acc + node.getAttribute('radius');
        }, 0),
        text: groupId,
        badges: {
          bottomRight: {
            text: nodes.size
          }
        },
        image: FLAGS[groupId]
      }
    };
  },
  duration,
  enabled: false
});

const btn = document.getElementById('group-btn');

btn.addEventListener('click', () => {

  // Disable the button while the transformation is running
  btn.disabled = true;

  // Toggle the grouping
  transformation.toggle(duration).then(() => {
    btn.disabled = false;
    ogma.layouts.force({
      locate: true
    });
  });
});

ogma.events.on('click', evt => {
  if (evt.target && evt.target.isNode) {
    const subNodes = evt.target.getData('subNodes');
    console.log(subNodes && subNodes.getId());
  }
});

// UI update
const onToggle = ({ target }) => {
  ogma.transformations.triggerGroupsUpdated();
  const enabled = target.isEnabled();
  const buttonText = enabled ? 'Ungroup nodes' : 'Group nodes';
  btn.textContent = buttonText;
};
ogma.events.on('transformationEnabled', onToggle);
ogma.events.on('transformationDisabled', onToggle);
```

## Tags

transformations, grouping

## See Also

- [Live Example](https://doc.linkurio.us/ogma/latest/examples/group-nodes.html)
- [API Reference](references/REFERENCE.md)
