# Force layout

This example shows how to use the [force-directed layout](/api/ogma/layouts.html#ogma-layouts-force) which is the most common layout used to position nodes in a graph.
`charge` defines the strength with which the nodes repel each other. `gravity` stands for the global force, pulling the graph to its mass center. `elasticity` defines how the nodes collide with each other.

## Code

```typescript
import Ogma, { LocateOptions, RawGraph } from '@linkurious/ogma';
import { random } from './randomize';

const ogma = new Ogma({
  container: 'graph-container',
  options: { interactions: { drag: { enabled: false } } }
});
const padding = 100;
const animationDuration = 300;

// disable node dragging
//ogma.setOptions({ interactions: { drag: { enabled: false }}});

const randomize = () => {
  ogma.getNodes().forEach(node => {
    node.setAttributes({
      x: random() * 150,
      y: random() * 150
    });
  });
};

// generate random graph
ogma.generate
  .barabasiAlbert({
    nodes: 500,
    m0: 10,
    m: 1
  })
  .then(graph => {
    // add some disconnected components and 'orphan' nodes
    graph.nodes.forEach(node => {
      delete node.attributes;
    });
    addSmallComponents(graph, Math.floor(Math.log(graph.nodes.length)), 5);
    addLooseNodes(graph, Math.floor(graph.nodes.length / 3));
    ogma.setGraph(graph);
    return ogma.view.locateGraph({ padding: 100 });
  })
  .then(() => {
    return run({ padding });
  });

const form = document.querySelector<HTMLFormElement>('#params')!;
let callTimer = 0;

let charge: number, gravity: number, elasticity: number, edgeStrength: number;
const update = () => {
  charge = parseFloat(form['charge'].value);
  gravity = parseFloat(form['gravity']!.value);
  elasticity = parseFloat(form['elasticity']!.value);
  edgeStrength = parseFloat(form['edgeStrength']!.value);

  document.getElementById('charge-value')!.innerHTML = charge.toString();
  document.getElementById('gravity-value')!.innerHTML = gravity.toString();
  document.getElementById('elasticity-value')!.innerHTML =
    elasticity.toString();
  document.getElementById('edgeStrength-value')!.innerHTML =
    edgeStrength.toString();
};

form.addEventListener(
  'input',
  () => {
    clearTimeout(callTimer);
    update();
    callTimer = setTimeout(run, 100) as unknown as number;
  },
  true
);

document.querySelector('#randomize')!.addEventListener('click', randomize);
document.querySelector('#run')!.addEventListener('click', () => {
  run();
});

const run = (locate?: LocateOptions) => {
  const start = Date.now();
  return ogma.layouts
    .force({
      //useWebWorker: false,
      charge: charge,
      gravity: gravity,
      elasticity: elasticity,
      locate: locate,
      duration: animationDuration,
      edgeStrength: edgeStrength
    })
    .then(() => {
      const duration = Date.now() - start;
      document.getElementById('time')!.innerHTML =
        'done in ' + ((duration - animationDuration) / 1000).toFixed(2) + 's';

      document.getElementById('loading')!.style.display = 'none';
    });
};

function addLooseNodes(graph: RawGraph, n: number) {
  const baseId = graph.nodes.length;
  for (let i = 0; i < n; i++) {
    graph.nodes.push({ id: baseId + i });
  }
}

function addSmallComponents(graph: RawGraph, n: number, m: number) {
  for (let i = 0; i < n; i++) {
    const baseId = graph.nodes.length;
    for (let j = 0; j < m + 1; j++) {
      graph.nodes.push({ id: baseId + j });
    }
    for (let k = 1; k < m + 1; k++) {
      graph.edges.push({ source: baseId, target: baseId + k });
    }
  }
}

update();
```

## Tags

layout, force-layout, algorithms, performance

## See Also

- [Live Example](https://doc.linkurio.us/ogma/latest/examples/layout-force.html)
- [API Reference](references/REFERENCE.md)
