# Grid

This example shows how to use the [grid layout](/api/ogma/layouts.html#ogma-layouts-grid) algorithm. The grid layout is a simple layout algorithm that places the nodes on a grid. It is useful for visualizing meshes.

## Code

```typescript
import Ogma from '@linkurious/ogma';
import { GUI } from '@linkurious/ogma-ui-kit/gui';
import { random } from './randomize';

const ogma = new Ogma({
  container: 'graph-container'
});

// Generate a random graph
const graph = await ogma.generate.random({ nodes: 100, edges: 200 });
// Assign it to Ogma
await ogma.setGraph(graph);
// Set a node size to order them by size.
await ogma.getNodes().setAttribute(
  'radius',
  ogma.getNodes().map(() => 2 + 10 * random())
);

// Apply a grid layout to the graph
await ogma.layouts.grid({
  rows: 8,
  cols: 8,
  sortBy: 'radius',
  colDistance: 5,
  rowDistance: 2,
  reverse: true,
  duration: 1500,
  locate: true
});

// @ts-expect-error
const gui = new GUI();
const options = {
  rows: 8,
  columns: 8,
  colDistance: 0,
  rowDistance: 0,
  sortBy: 'biggest to smallest',
  run: () => {
    console.log('run', options.sortBy);

    const sortBy =
      options.sortBy === 'biggest to smallest' ||
      options.sortBy === 'smallest to biggest'
        ? 'radius'
        : options.sortBy === 'degree'
          ? 'degree'
          : options.sortBy;
    const reverse = options.sortBy === 'biggest to smallest';

    console.log({ rows: options.rows, columns: options.columns });

    ogma.layouts.grid({
      rows: options.rows,
      cols: options.columns,
      sortBy,
      reverse,
      sortFallbackValue: 0,
      // undefined used here to avoid the 0 value to be passed to the layout
      // since it doesn't make sense
      rowDistance: options.rowDistance || undefined,
      colDistance: options.colDistance || undefined,
      locate: true,
      duration: 1500
    });
  }
};
gui
  .add(options, 'rows', 2, 20)
  .step(1)
  .onChange(() => {
    options.columns = Math.floor(ogma.getNodes().size / options.rows);
    options.run();
  });
gui
  .add(options, 'columns', 2, 20)
  .step(1)
  .onChange(() => {
    options.rows = Math.floor(ogma.getNodes().size / options.columns);
    options.run();
  });
gui.add(options, 'colDistance', 0, 200).step(1).onChange(options.run);
gui.add(options, 'rowDistance', 0, 200).step(1).onChange(options.run);
gui
  .add(options, 'sortBy', [
    'biggest to smallest',
    'smallest to biggest',
    'degree',
    'random'
  ])
  .onChange(options.run);

gui.add(options, 'run').name('Run layout');
```

## Tags

layout, algorithms

## See Also

- [Live Example](https://doc.linkurio.us/ogma/latest/examples/layout-grid.html)
- [API Reference](references/REFERENCE.md)
