# Icon

This example shows how to use the [icon attribute](/api/types/nodeattributes.html) to display an icon on a node.
Icons are defined as font glyphs. Ogma works well with
<a href="https://fontawesome.io/" target="_blank" rel="noopener">FontAwesome</a>,
but any font such as Arial may be used.
<br />
A node's icon is controlled by the <strong>icon.content</strong> attribute. The
font is defined by the <strong>icon.font</strong> attribute, the color by the
<strong>icon.color</strong> and size relative to the node size in the
<strong>icon.scale</strong>. Icons can also be used in badges as illustrated in
the <a href="node-badge.html">badge example</a>.

## Code

```typescript
import Ogma from '@linkurious/ogma';
import chroma from 'chroma-js';

// dummy icon element to retrieve the HEX code
const placeholder = document.getElementById('icon-placeholder')!;
placeholder.style.visibility = 'hidden';

const CLEAN_QUOTES = /["']/g;
// helper routine to get the icon HEX code
const getIconCode = (icon: string) => {
  placeholder.className = `icon-${icon}`;
  const content = getComputedStyle(placeholder, ':before').content;
  return content.replace(CLEAN_QUOTES, '');
};

const fontName = 'Lucide';

const ICONS = [
  'shield',
  'chevron-down',
  'folder',
  'target',
  'bug',
  'martini',
  'star',
  'glasses',
  'cloud',
  'arrow-left',
  'user',
  'message-square',
  'play',
  'hammer'
];

const getColor = chroma
  .scale(['0093ff', 'ff0000'])
  .domain([0, ICONS.length + 2]);

const ogma = new Ogma({
  container: 'graph-container'
});

// Specify the style.
ogma.styles.addNodeRule({
  // Specify the icon font, color and contents.
  icon: {
    font: fontName,
    color: n => getColor(+n.getId()).hex(),
    content: n => getIconCode(ICONS[+n.getId() % ICONS.length]),
    scale: 0.5
  },
  text: {
    content: n => ICONS[+n.getId() % ICONS.length],
    backgroundColor: n => getColor(+n.getId()).hex(),
    color: 'white',
    font: 'IBM Plex Sans',
    size: 14,
    tip: false
  },
  // Set the node color as white
  color: 'white',
  // Add a border to the node
  innerStroke: {
    color: n => getColor(+n.getId()).hex(),
    width: 3,
    minVisibleSize: 5
  }
});

ogma.styles.setHoveredNodeAttributes({
  text: {
    backgroundColor: n => n.getAttribute('icon.color')
  }
});

ogma.generate
  .random({ nodes: ICONS.length + 2, edges: 0 })
  .then(graph => ogma.setGraph(graph))
  .then(() => ogma.layouts.grid({ locate: true, duration: 0 }));
```

## Tags

styles, images, icons

## See Also

- [Live Example](https://doc.linkurio.us/ogma/latest/examples/node-icon.html)
- [API Reference](references/REFERENCE.md)
