# Visual grouping

This example shows how to use [node grouping](/api/ogma/transformations.html#ogma-transformations-addnodegrouping) to group nodes and show their content. Double click on a group to open/close it.

## Code

```typescript
import Ogma from '@linkurious/ogma';
import { backgroundColor, FLAGS, duration } from './const';
import { addStyles } from './styles';
import { NodeData, Country } from './types';

const ogma = new Ogma<NodeData>({
  container: 'graph-container',
  options: {
    backgroundColor
  }
});
addStyles(ogma);

Ogma.parse.jsonFromUrl<NodeData>('countries.json').then(graph => {
  ogma.addGraph(graph, { ignoreInvalid: true });
});

const byCountry = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => node.getData('location'),
  nodeGenerator: (nodes, country) => {
    const isVisible = country !== 'Japan';
    return {
      id: 'special group ' + country,
      data: {
        country: country,
        open: isVisible
      },
      attributes: {
        text: country,
        // fixed radius for the nodes with hidden contents
        radius: 20,
        image: FLAGS[country as Country]
      }
    };
  },
  showContents: node => node.getData('open'),
  onGroupUpdate: (metaNode, subNodes, open) => {
    const groupId = metaNode.getId();
    if (!layoutPerGroup.has(groupId)) {
      layoutPerGroup.set(groupId, defaultLayout);
    }
    if (!open) return;
    if (layoutPerGroup.get(groupId) === 'hierarchical') {
      return {
        layout: 'hierarchical'
      };
    }
    return {
      layout: 'force'
    };
  },
  duration,
  enabled: false
});
const layoutPerGroup = new Map();
const defaultLayout = 'force';
const customGroup = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => {
    const id = node.getData('country');
    if (id === 'France' || id === 'Brazil') return 'Level 2';
    return undefined;
  },
  nodeGenerator: (nodes, country) => ({
    id: 'special group ' + country,
    data: {
      country: country,
      open: true
    },
    attributes: {
      color: backgroundColor,
      text: country,
      opacity: 0.32
    }
  }),
  onGroupUpdate: () => {
    return {
      layout: 'force'
    };
  },
  showContents: node => node.getData('open'),
  enabled: false,
  duration
});
// UI buttons
const buttonFranceBrazilOnly = document.getElementById(
  'custom-group-btn'
) as HTMLButtonElement;
const buttonSwitchLayout = document.getElementById(
  'switch-layout'
) as HTMLButtonElement;
const buttonGroupAll = document.getElementById(
  'group-btn'
) as HTMLButtonElement;

const disableAllButtons = () => {
  buttonGroupAll.disabled = true;
  buttonFranceBrazilOnly.disabled = true;
  buttonSwitchLayout.disabled = true;
};

buttonFranceBrazilOnly.disabled = true;
buttonSwitchLayout.disabled = true;

buttonGroupAll.addEventListener('click', () => {
  // Disable the button to prevent multiple clicks
  disableAllButtons();

  // Trigger global layout
  ogma.transformations.triggerGroupsUpdated();

  // Toggle the grouping transformation
  byCountry.toggle().then(() => {
    const buttonText = byCountry!.isEnabled()
      ? 'Ungroup by country'
      : 'Group by country';
    buttonGroupAll.textContent = buttonText;
    buttonGroupAll.disabled = false;
    buttonFranceBrazilOnly.disabled = ! byCountry.isEnabled();
    buttonSwitchLayout.disabled = ! byCountry.isEnabled();
  });
});

buttonFranceBrazilOnly.addEventListener('click', () => {

  // Disable the button to prevent multiple clicks
  disableAllButtons();

  // Toggle the grouping
  ogma.transformations.triggerGroupsUpdated();
  customGroup.toggle().then(() => {
    const buttonText = customGroup!.isEnabled()
      ? 'Ungroup France & Brazil'
      : 'Group France & Brazil';

    buttonFranceBrazilOnly.textContent = buttonText;
    buttonGroupAll.disabled = false;
    buttonFranceBrazilOnly.disabled = false;
    buttonSwitchLayout.disabled = ! customGroup.isEnabled();
  });
});

buttonSwitchLayout.addEventListener('click', () => {
  const groups = byCountry.getContext().metaNodes;
  // setup the layout for each group
  groups.getId().forEach(id => {
    layoutPerGroup.set(id, 'hierarchical');
  });
  // tell transformations to rerun the group update callback
  ogma.transformations.triggerGroupUpdated(groups);
  // trigger the refresh
  byCountry.refresh();
});

ogma.events
  .on('click', evt => {
    const target = evt.target;
    if (target && target.isNode && target.isVirtual()) {
      console.log('sub nodes', target.getSubNodes()!.getId());
    }
  })
  .on('doubleclick', evt => {
    const target = evt.target;
    if (target && target.isNode && target.isVirtual()) {
      target.setData('open', !target.getData('open'));
    }
  })
  .on('transformationDisabled', () => {
    buttonSwitchLayout.disabled = !ogma.transformations
      .getList()
      .some(e => e.isEnabled());
  })
  .on('transformationEnabled', () => {
    buttonSwitchLayout.disabled = false;
  });

ogma.transformations.onGroupsUpdated(() => ({
  layout: 'force'
}));
```

## Tags

transformations, grouping, visual-grouping

## See Also

- [Live Example](https://doc.linkurio.us/ogma/latest/examples/visual-grouping.html)
- [API Reference](references/REFERENCE.md)
