# Core API

Graph data management in Ogma: nodes, edges, collections, and data access.

## Graph Structure

Ogma stores an internal graph consisting of:
- **Nodes**: Entities with unique IDs, optional data, and visual attributes
- **Edges**: Relationships connecting nodes, with source/target references

## Data Format (RawGraph)

```typescript
interface RawGraph {
  nodes: RawNode[];
  edges: RawEdge[];
}

interface RawNode {
  id: NodeId;                    // string | number (required)
  data?: Record<string, any>;    // Custom properties
  attributes?: NodeAttributes;   // Visual styling
}

interface RawEdge {
  id?: EdgeId;                   // Optional, auto-generated if omitted
  source: NodeId;                // Source node ID (required)
  target: NodeId;                // Target node ID (required)
  data?: Record<string, any>;    // Custom properties
  attributes?: EdgeAttributes;   // Visual styling
}
```

## Loading Data

```typescript
// Set entire graph (clears existing)
await ogma.setGraph({
  nodes: [{ id: 'n1' }, { id: 'n2' }],
  edges: [{ source: 'n1', target: 'n2' }]
});

// Add to existing graph
await ogma.addGraph({
  nodes: [{ id: 'n3' }],
  edges: [{ source: 'n1', target: 'n3' }]
});

// Parse from URL
const graph = await Ogma.parse.jsonFromUrl('data/graph.json');
await ogma.setGraph(graph);

// Clear graph
ogma.clearGraph();
```

## Adding Nodes and Edges

```typescript
// Single node
const node = ogma.addNode({
  id: 'user1',
  data: { name: 'Alice', role: 'admin' }
});

// Multiple nodes (batch - more performant for >1000 items)
// ALWAYS prefer batch methods for large additions
await ogma.addNodes([
  { id: 'user2', data: { name: 'Bob' } },
  { id: 'user3', data: { name: 'Charlie' } }
]);

// Single edge
const edge = ogma.addEdge({
  id: 'e1',
  source: 'user1',
  target: 'user2',
  data: { relationship: 'manages' }
});

// Multiple edges
await ogma.addEdges([
  { source: 'user1', target: 'user3' },
  { source: 'user2', target: 'user3' }
]);
```

## Accessing Nodes and Edges

```typescript
// Get by ID
const node = ogma.getNode('user1');           // Node | null
const edge = ogma.getEdge('e1');              // Edge | null

// Get multiple by IDs
const nodes = ogma.getNodes(['user1', 'user2']); // NodeList

// Get all
const allNodes = ogma.getNodes();             // NodeList
const allEdges = ogma.getEdges();             // EdgeList

// Check existence
if (ogma.getNode('user1')) { /* exists */ }
```

## Node Object

```typescript
const node = ogma.getNode('user1');

// Identity
node.getId();                    // NodeId
node.isNode;                     // true (type guard)

// Data access
node.getData();                  // All data as object
node.getData('name');            // Specific property
node.getData('address.city');    // Nested property (dot notation)

// Data modification
node.setData('name', 'Alice Smith');
node.setData({ name: 'Alice', role: 'user' }); // Replace all data

// Topology
node.getDegree();                // Total connections count
node.getAdjacentNodes();         // NodeList of neighbors
node.getAdjacentEdges();         // EdgeList of connected edges
node.getConnectedNodes();        // Same as getAdjacentNodes

// Position
node.getPosition();              // { x: number, y: number }
node.setPosition(100, 200);
```

## Edge Object

```typescript
const edge = ogma.getEdge('e1');

// Identity
edge.getId();                    // EdgeId
edge.isNode;                     // false
edge.isEdge;                     // true (type guard)

// Endpoints
edge.getSource();                // Node (source node)
edge.getTarget();                // Node (target node)
edge.getExtremities();           // NodeList [source, target]

// Data access (same as Node)
edge.getData();
edge.getData('relationship');
edge.setData('weight', 10);
```

## NodeList and EdgeList

Collections provide batch operations and functional methods.

```typescript
const nodes = ogma.getNodes();

// Size
nodes.size;                      // Number of elements
nodes.isEmpty();                 // boolean

// Iteration
nodes.forEach(node => { /* do something with the node */ });
nodes.map(node => node.getData('name'));      // any[]
nodes.filter(node => node.getDegree() > 5);   // NodeList
nodes.find(node => node.getData('role') === 'admin'); // Node | undefined
nodes.reduce((acc, node) => acc + 1, 0); // number
nodes.inverse();                 // NodeList (all nodes NOT in this list)

// Batch data access
nodes.getId();                   // NodeId[]
nodes.getData();                 // object[] (all data)
nodes.getData('name');           // any[] (specific property)

// Batch modification
// ALWAYS prefer batch methods for performance
nodes.setData('visited', true);
nodes.setAttributes({ color: 'red' });

// Set operations
nodes.includes(node);            // boolean
nodes.concat(otherNodes);        // NodeList (combine)
nodes.filter(predicate);         // NodeList (subset)
nodes.inverse();                 // NodeList (all nodes NOT in this list)

// Conversion
nodes.toArray();                 // Node[]
nodes.toList();                  // NodeList (copy)

// Get connected elements
nodes.getAdjacentNodes();        // NodeList (all neighbors)
nodes.getAdjacentEdges();        // EdgeList (all connected edges)
```

## Removing Elements

```typescript
// Remove single
ogma.removeNode('user1');
ogma.removeEdge('e1');

// Remove multiple
// ALWAYS prefer batch methods for performance
ogma.removeNodes(['user1', 'user2']);
ogma.removeEdges(['e1', 'e2']);

// Remove from collection
const nodesToRemove = ogma.getNodes().filter(n => n.getData('temporary'));
ogma.removeNodes(nodesToRemove);

// Clear all
ogma.clearGraph();
```

## Graph Traversal

```typescript
// Get neighbors
const neighbors = ogma.getNode('center').getAdjacentNodes();

// Filter by edge type
const friends = ogma.getNode('user1')
  .getAdjacentEdges()
  .filter(e => e.getData('type') === 'friend')
  .getSource()                   // or getTarget()
  .concat(
    ogma.getNode('user1')
      .getAdjacentEdges()
      .filter(e => e.getData('type') === 'friend')
      .getTarget()
  );

// BFS/DFS traversal
ogma.algorithms.bfs({
  source: ogma.getNode('root'),
  onNode: node => console.log(node.getId())
});
```

## Type Safety

Use generics for typed data access:

```typescript
interface NodeData {
  name: string;
  category: 'person' | 'company';
  revenue?: number;
}

interface EdgeData {
  relationship: string;
  weight: number;
}

const ogma = new Ogma<NodeData, EdgeData>({
  container: 'graph-container'
});

// Now getData() returns typed values
const node = ogma.getNode('n1');
const name: string = node.getData('name');       // TypeScript knows type
const category = node.getData('category');       // 'person' | 'company'
```

## Performance Tips

1. **Batch operations**: Use `addNodes`/`addEdges` instead of individual `addNode`/`addEdge` for >100 items
2. **Avoid repeated lookups**: Cache `ogma.getNode()` results when accessing the same node multiple times
3. **Use NodeList methods**: Prefer `nodeList.setData()` over `nodeList.forEach(n => n.setData())`
4. **Progressive loading**: Use `batchSize` option for very large graphs:
   ```typescript
   await ogma.addGraph(largeGraph, { batchSize: 1000 });
   ```
