# Events

Event handling and user interactions in Ogma: clicks, hover, selection, drag, and keyboard.

## Event System Basics

```typescript
// Subscribe to event
ogma.events.on('click', handler);

// Unsubscribe
ogma.events.off('click', handler);

// One-time listener
ogma.events.once('click', handler);
```

## Mouse/Touch Events

### Click

```typescript
ogma.events.on('click', (event) => {
  const { target, x, y, button } = event;

  if (target === null) {
    // Clicked on background
    console.log('Background click at', x, y);
  } else if (target.isNode) {
    // Clicked on node
    console.log('Node clicked:', target.getId());
  } else if (target.isEdge) {
    // Clicked on edge
    console.log('Edge clicked:', target.getId());
  }
});

// Right-click
ogma.events.on('click', (event) => {
  if (event.button === 'right' && event.target?.isNode) {
    showContextMenu(event.target, event.x, event.y);
  }
});
```

### Double Click

```typescript
ogma.events.on('doubleclick', (event) => {
  if (event.target?.isNode) {
    // Expand node on double-click
    expandNode(event.target);
  }
});
```

### Hover (Mouse Over/Out)

```typescript
ogma.events.on('mouseover', (event) => {
  if (event.target?.isNode) {
    event.target.setAttributes({
      outerStroke: { color: 'yellow', width: 3 }
    });
  }
});

ogma.events.on('mouseout', (event) => {
  if (event.target?.isNode) {
    event.target.resetAttributes(['outerStroke']);
  }
});

// Get currently hovered element
const hovered = ogma.getHoveredElement();  // Node | Edge | null
```

### Mouse Move

```typescript
ogma.events.on('mousemove', (event) => {
  // Runs continuously as mouse moves
  updateTooltipPosition(event.x, event.y);
});
```

## Drag Events

### Node Dragging

```typescript
ogma.events.on('nodesDragStart', (event) => {
  console.log('Started dragging', event.nodes.size, 'nodes');
  event.nodes.addClass('dragging');
});

ogma.events.on('nodesDragProgress', (event) => {
  // Runs continuously during drag
  console.log('Dragging at', event.x, event.y);
});

ogma.events.on('nodesDragEnd', (event) => {
  event.nodes.removeClass('dragging');
  console.log('Finished dragging');
});
```

### Disable Node Dragging

```typescript
// Globally
ogma.setOptions({ interactions: { drag: { enabled: false } } });

// Per node
node.setAttributes({ draggable: false });
```

## Selection Events

### Selection Changes

```typescript
ogma.events.on('nodesSelected', (event) => {
  console.log('Selected nodes:', event.nodes.getId());
});

ogma.events.on('nodesUnselected', (event) => {
  console.log('Unselected nodes:', event.nodes.getId());
});

ogma.events.on('edgesSelected', (event) => {
  console.log('Selected edges:', event.edges.getId());
});

ogma.events.on('edgesUnselected', (event) => {
  console.log('Unselected edges:', event.edges.getId());
});
```

### Selection API

```typescript
// Get current selection
const selectedNodes = ogma.getSelectedNodes();  // NodeList
const selectedEdges = ogma.getSelectedEdges();  // EdgeList

// Set selection programmatically
ogma.getNodes(['n1', 'n2']).setSelected(true);
node.setSelected(true);

// Clear selection
ogma.clearSelection();

// Get non-selected
const unselected = ogma.getNonSelectedNodes();

// Invert selection
const current = ogma.getSelectedNodes();
ogma.getNonSelectedNodes().setSelected(true);
current.setSelected(false);
```

### Custom Selection Behavior

```typescript
// Disable default selection
ogma.setOptions({ interactions: { selection: { enabled: false } } });

// Implement custom selection
ogma.events.on('click', (event) => {
  if (event.target?.isNode) {
    // Toggle selection on click
    event.target.setSelected(!event.target.isSelected());
  } else {
    // Clear on background click
    ogma.clearSelection();
  }
});
```

## Selection Tools

### Lasso Selection

```typescript
// Enable lasso tool
ogma.tools.lasso.enable();

// Check if enabled
ogma.tools.lasso.enabled();  // boolean

// Disable
ogma.tools.lasso.disable();

// Listen for lasso completion
ogma.events.on('nodesSelected', () => {
  console.log('Lasso selected:', ogma.getSelectedNodes().getId());
});
```

### Rectangle Selection

```typescript
ogma.tools.rectangleSelect.enable();

// Enable on Ctrl+drag
ogma.events.on('dragStart', () => {
  if (ogma.keyboard.isKeyPressed('ctrl')) {
    ogma.tools.rectangleSelect.enable();
  }
});
```

## Keyboard Events

```typescript
// Key press handler
ogma.events.onKeyPress('Delete', () => {
  ogma.removeNodes(ogma.getSelectedNodes());
});

// Multiple keys
ogma.events
  .onKeyPress('l', () => ogma.tools.lasso.enable())
  .onKeyPress('Escape', () => {
    ogma.tools.lasso.disable();
    ogma.clearSelection();
  });

// Zoom with +/-
ogma.events
  .onKeyPress('+', () => ogma.view.zoomIn({ duration: 200 }))
  .onKeyPress('-', () => ogma.view.zoomOut({ duration: 200 }));

// Check if key is pressed
if (ogma.keyboard.isKeyPressed('ctrl')) { /* ... */ }
if (ogma.keyboard.isKeyPressed('shift')) { /* ... */ }
```

## View/Camera Events

```typescript
// Zoom events
ogma.events.on('zoomStart', () => console.log('Zoom starting'));
ogma.events.on('zoomProgress', ({ zoom }) => console.log('Zoom:', zoom));
ogma.events.on('zoomEnd', () => console.log('Zoom finished'));

// Pan events
ogma.events.on('panStart', () => console.log('Panning'));
ogma.events.on('panEnd', () => console.log('Pan finished'));

// General view change (after zoom or pan settles)
ogma.events.on('viewChanged', () => {
  console.log('View changed');
  updateMinimap();
});
```

## Graph Events

```typescript
// Nodes added/removed
ogma.events.on('nodesAdded', ({ nodes }) => {
  console.log('Added', nodes.size, 'nodes');
});

ogma.events.on('nodesRemoved', ({ nodes }) => {
  console.log('Removed', nodes.size, 'nodes');
});

// Edges added/removed
ogma.events.on('edgesAdded', ({ edges }) => { /* ... */ });
ogma.events.on('edgesRemoved', ({ edges }) => { /* ... */ });

// Data changes
ogma.events.on('nodesDataUpdated', ({ nodes }) => {
  console.log('Node data updated:', nodes.getId());
});

// Attributes changes
ogma.events.on('nodesAttributesUpdated', ({ nodes }) => { /* ... */ });
```

## Layout Events

```typescript
ogma.events.on('layoutStart', ({ name }) => {
  showSpinner();
});

ogma.events.on('layoutEnd', ({ name }) => {
  hideSpinner();
});

ogma.events.on('layoutProgress', ({ progress, name }) => {
  updateProgressBar(progress);
});
```

## Coordinate Conversion

```typescript
// Graph coordinates (where nodes live) to screen pixels
const screenPos = ogma.view.graphToScreenCoordinates({ x: 100, y: 50 });
showTooltipAt(screenPos.x, screenPos.y);

// Screen pixels to graph coordinates
const graphPos = ogma.view.screenToGraphCoordinates({ x: event.clientX, y: event.clientY });
```

## Common Patterns

### Tooltip on Hover

```typescript
const tooltip = document.getElementById('tooltip');

ogma.events.on('mouseover', (event) => {
  if (event.target?.isNode) {
    const pos = ogma.view.graphToScreenCoordinates(event.target.getPosition());
    tooltip.style.left = pos.x + 'px';
    tooltip.style.top = pos.y + 'px';
    tooltip.textContent = event.target.getData('name');
    tooltip.style.display = 'block';
  }
});

ogma.events.on('mouseout', () => {
  tooltip.style.display = 'none';
});
```

### Context Menu

```typescript
ogma.events.on('click', (event) => {
  if (event.button === 'right' && event.target?.isNode) {
    event.preventDefault?.();
    const pos = ogma.view.graphToScreenCoordinates({ x: event.x, y: event.y });
    showContextMenu(event.target, pos.x, pos.y);
  }
});
```

### Highlight Neighbors on Hover

```typescript
const neighborClass = ogma.styles.createClass({
  name: 'neighbor',
  nodeAttributes: { outerStroke: { color: 'blue', width: 2 } },
  edgeAttributes: { color: 'blue', width: 2 }
});

ogma.events.on('mouseover', (event) => {
  if (event.target?.isNode) {
    event.target.getAdjacentNodes().addClass('neighbor');
    event.target.getAdjacentEdges().addClass('neighbor');
  }
});

// Clear highlights on mouse out
// ALWAYS prefer 'mousemove' event for hit testing
ogma.events.on('mouseout', (event) => {
  if (event.target?.isNode) {
    neighborClass.clearNodes();
    neighborClass.clearEdges();
  }
});
```

## React Integration

```tsx
import { Ogma, useOgma } from '@linkurious/ogma-react';
import { useEffect } from 'react';

function EventHandler() {
  const ogma = useOgma();

  useEffect(() => {
    const handleClick = (event) => {
      if (event.target?.isNode) {
        console.log('Clicked:', event.target.getId());
      }
    };

    ogma.events.on('click', handleClick);
    return () => ogma.events.off('click', handleClick);
  }, [ogma]);

  return null;
}

function App() {
  return (
    <Ogma graph={graph}>
      <EventHandler />
    </Ogma>
  );
}
```

## Vue Integration

```vue
<script setup lang="ts">
import Ogma from '@linkurious/ogma';
import { ref, onMounted, onUnmounted } from 'vue';

const ogma = ref<Ogma | null>(null);

const handleClick = (event) => {
  if (event.target?.isNode) {
    console.log('Clicked:', event.target.getId());
  }
};

onMounted(() => {
  ogma.value = new Ogma({ container: 'graph' });
  ogma.value.events.on('click', handleClick);
});

onUnmounted(() => {
  ogma.value?.events.off('click', handleClick);
  ogma.value?.destroy();
});
</script>
```
