# Layouts

Graph layout algorithms in Ogma: force-directed, hierarchical, radial, and more.

## Overview

Layouts automatically position nodes to reveal graph structure. All layout methods return Promises and can be awaited.

```typescript
await ogma.layouts.force();
await ogma.view.locateGraph();  // Center view on result
```

## Force Layout

Best for: General-purpose visualization, social networks, organic structures.

Simulates physical forces: nodes repel each other, edges act as springs.

```typescript
// Basic usage
await ogma.layouts.force();

// With options
await ogma.layouts.force({
  charge: 20,              // Node repulsion strength (default: 10)
  gravity: 0.01,           // Pull toward center (default: 0.01)
  elasticity: 0.3,         // Edge spring strength (default: 0.1)
  steps: 300,              // Simulation iterations (default: 300)
  duration: 500,           // Animation duration in ms (0 = instant)
  locate: true             // Center view after layout
});
```

### GPU-Accelerated Force Layout

For large graphs (10k+ nodes), use GPU acceleration:

```typescript
await ogma.layouts.force({
  gpu: true,
  steps: 500
});
```

### Incremental Force Layout

Apply force layout to newly added nodes while keeping existing positions stable:

```typescript
// Pin existing nodes, only layout new ones
await ogma.layouts.force({
  nodes: newNodes,           // Only layout these nodes
  incremental: true          // Keep other nodes in place
});
```

## Hierarchical Layout

Best for: Trees, DAGs, org charts, dependency graphs.

Arranges nodes in levels based on directed relationships.

```typescript
// Basic usage
await ogma.layouts.hierarchical();

// With options
await ogma.layouts.hierarchical({
  direction: 'TB',           // 'TB' | 'BT' | 'LR' | 'RL' (top-bottom, etc.)
  nodeDistance: 50,          // Min distance between nodes in same level
  levelDistance: 100,        // Distance between levels
  duration: 400,
  locate: true
});
```

### Custom Root Nodes

```typescript
await ogma.layouts.hierarchical({
  roots: ['node1', 'node2'],   // Force these nodes to top level
  sinks: ['node5']             // Force these to bottom level
});
```

### Layer Constraints

Constrain specific nodes to specific levels via data:

```typescript
// In node data
{ id: 'ceo', data: { layer: 0 } }      // Top level
{ id: 'manager', data: { layer: 1 } }  // Second level
{ id: 'employee', data: { layer: 2 } } // Third level

await ogma.layouts.hierarchical();  // Respects layer constraints
```

## Sequential Layout

Best for: Pipelines, workflows, data flows.

Similar to hierarchical but optimizes for horizontal alignment.

```typescript
await ogma.layouts.sequential({
  direction: 'LR',           // Usually left-to-right
  nodeDistance: 40,
  levelDistance: 80
});
```

## Radial Layout

Best for: Ego networks, centered exploration, tree-like structures.

Arranges nodes in concentric circles around a center node.

```typescript
// Around specific center node
await ogma.layouts.radial({
  centerNode: ogma.getNode('root'),
  nodeDistance: 30,
  levelDistance: 80,
  duration: 500
});

// Around current selection
await ogma.layouts.radial({
  centerNode: ogma.getSelectedNodes().get(0)
});
```

## Grid Layout

Best for: Uniform display, comparison views, matrix layouts.

```typescript
await ogma.layouts.grid({
  columns: 10,               // Number of columns (or rows)
  nodeDistance: 50,          // Spacing between nodes
  locate: true
});
```

## Concentric Layout

Best for: Centrality visualization, importance-based layouts.

Arranges nodes in circles based on a metric.

```typescript
await ogma.layouts.concentric({
  centralNode: ogma.getNode('center'),
  levelDistance: 60
});
```

## ForceLink Layout

Older version of the force-directed layout. Very good for small graphs but less performant and more complex to adjust than the main `force` layout.

```typescript
// Basic usage
await ogma.layouts.forceLink();

// With options
await ogma.layouts.forceLink({
  scalingRatio: 100,           // Distance factor between nodes (higher = more spread)
  gravity: 1,                  // Attraction to center (higher = more compact)
  edgeWeightInfluence: 0,      // Attraction based on edge weights (0 = disabled)
  strongGravityMode: true,     // Strong gravity formula
  slowDown: 1,                 // Reduces speed as iterations increase
  autoStop: true,              // Stop automatically when stable
  maxIterations: 1000,         // Max iterations if autoStop is true
  avgDistanceThreshold: 0.01,  // Stopping threshold for autoStop
  barnesHutOptimize: false,    // Barnes-Hut optimization for large graphs (5000+ nodes)
  barnesHutTheta: 0.5,         // Theta for Barnes-Hut
  linLogMode: false,           // Alternative energy model
  alignNodeSiblings: true,     // Align nodes linked to same two nodes
  nodeSiblingsScale: 5,        // Distance between aligned siblings
  locate: true,                // Smooth transition to result
  duration: 300
});

// Custom node mass and edge weight
await ogma.layouts.forceLink({
  nodeMass: (node) => node.getDegree(),
  edgeWeight: (edge) => edge.getData('weight') || 1
});
```

## Local Layouts

Apply layout to a subset of nodes.

```typescript
// Layout only selected nodes
await ogma.layouts.force({
  nodes: ogma.getSelectedNodes()
});

// Layout neighborhood of a node
const center = ogma.getNode('focus');
const neighborhood = center.getAdjacentNodes().concat(center);
await ogma.layouts.force({
  nodes: neighborhood,
  locate: { nodes: neighborhood }  // Center view on result
});
```

## Layout Events

```typescript
ogma.events.on('layoutStart', ({ name }) => {
  console.log(`Layout ${name} started`);
});

ogma.events.on('layoutEnd', ({ name }) => {
  console.log(`Layout ${name} finished`);
});

ogma.events.on('layoutProgress', ({ progress }) => {
  console.log(`Layout progress: ${progress}%`);
});
```

## Overlap Removal

Remove node overlaps after any layout:

```typescript
await ogma.layouts.force();
await ogma.tools.removeOverlap({ margin: 5 });
```

## Layout with Grouping

When using visual grouping, layout group contents separately:

```typescript
const grouping = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => node.getData('cluster'),
  showContents: true,
  onGroupUpdate: async (metaNode, subNodes) => {
    // Layout contents of each group
    await ogma.layouts.force({ nodes: subNodes });
  }
});

// Layout the meta-nodes (groups themselves)
await grouping.whenApplied();
await ogma.layouts.force();
```

## Common Patterns

### Layout After Data Load

```typescript
const graph = await Ogma.parse.jsonFromUrl('data.json');
await ogma.setGraph(graph);
await ogma.layouts.force({ locate: true });
```

### Animated Layout Transition

```typescript
// Smooth transition to new layout
await ogma.layouts.hierarchical({ duration: 800, easing: 'cubicOut' });
```

### Layout Selection Only

```typescript
// Rearrange only selected nodes
ogma.events.on('keydown', async (e) => {
  if (e.key === 'l') {
    await ogma.layouts.force({
      nodes: ogma.getSelectedNodes(),
      incremental: true
    });
  }
});
```

## React Integration

```tsx
import { Ogma, useOgma } from '@linkurious/ogma-react';

function LayoutButtons() {
  const ogma = useOgma();

  return (
    <>
      <button onClick={() => ogma.layouts.force({ locate: true })}>
        Force Layout
      </button>
      <button onClick={() => ogma.layouts.hierarchical({ locate: true })}>
        Hierarchical
      </button>
    </>
  );
}
```

## Good Practices

### Smooth Transitions with `locate: true`

Always use `locate: true` in layout calls for smoother transitions - it centers the view on the result after layout completes:

```typescript
await ogma.layouts.force({ locate: true });
await ogma.layouts.hierarchical({ direction: 'LR', locate: true });
```

If locate option was used, avoid calling `ogma.view.locateGraph()` again as it is redundant.

### Performance Tips

1. **Use GPU** for force layout on large graphs: `{ gpu: true }`
2. **Reduce steps** for quick preview: `{ steps: 50 }`
3. **Skip animation** for instant result: `{ duration: 0 }`
4. **Use auto-stop** to halt when stable: `{ autoStop: true }`
5. **Layout incrementally** when adding nodes to existing layout
6. **Use `locate: true`** for automatic view centering after layout
