# Styles

Visual styling of nodes and edges in Ogma: attributes, rules, and classes.

## CSS-Like Styling Model

Ogma's styling system works similarly to CSS:

- **Style rules** are like CSS selectors and declarations - they match elements based on conditions and apply styles automatically
- **Classes** are like CSS classes - reusable named style sets you can add/remove from elements
- **Themes** are like CSS custom properties or design tokens - they define the default visual values for the entire visualization

This model allows you to separate styling concerns: define base appearance with themes, add dynamic styling with rules, and toggle visual states with classes.

## Styling Priority (Low to High)

1. **Original attributes** - Set when node/edge is created
2. **Style rules** - Dynamic rules applied to matching elements
3. **Individual attributes** - Set via `setAttributes()`
4. **Custom classes** - User-defined highlight states
5. **Built-in classes** - Selection and hover (always highest)

## Node Attributes

```typescript
interface NodeAttributes {
  // Position
  x?: number;
  y?: number;

  // Size and shape
  radius?: number;                        // Default: 5
  shape?: 'circle' | 'square' | 'diamond' | 'star' | 'pentagon' | 'hexagon';

  // Color
  color?: Color;                          // Fill color
  opacity?: number;                       // 0-1

  // Border
  stroke?: { color?: Color; width?: number };
  outerStroke?: { color?: Color; width?: number };

  // Text label
  text?: {
    content?: string;
    color?: Color;
    size?: number;
    font?: string;
    position?: 'center' | 'top' | 'bottom' | 'left' | 'right';
    backgroundColor?: Color;
  };

  // Icon
  icon?: {
    content?: string;                     // Unicode character or icon code
    font?: string;                        // Font family (e.g., 'FontAwesome')
    color?: Color;
    scale?: number;
  };

  // Image
  image?: {
    url?: string;
    scale?: number;
    fit?: 'cover' | 'contain';
  };

  // Effects
  halo?: { color?: Color; width?: number };
  badges?: Badge[];
  pulse?: { enabled?: boolean; color?: Color };
}
```

## Edge Attributes

```typescript
interface EdgeAttributes {
  // Size
  width?: number;                         // Default: 1

  // Color
  color?: Color;
  opacity?: number;

  // Shape
  shape?: 'line' | 'arrow' | 'tapered' | 'dotted' | 'dashed';

  // Text label
  text?: {
    content?: string;
    color?: Color;
    size?: number;
    backgroundColor?: Color;
  };

  // Effects
  halo?: { color?: Color; width?: number };
  pulse?: { enabled?: boolean };
}

type Color = string;  // CSS color: 'red', '#ff0000', 'rgb(255,0,0)', 'rgba(255,0,0,0.5)'
```

## Setting Attributes Directly

```typescript
// Single attribute
node.setAttribute('color', 'blue');
node.setAttribute('text.content', 'Label');

// Multiple attributes
node.setAttributes({
  color: 'red',
  radius: 10,
  text: { content: 'Node 1', color: 'white' }
});

// With animation
node.setAttributes(
  { color: 'green', radius: 15 },
  { duration: 500, easing: 'quadraticOut' }
);

// Reset to original
node.resetAttributes();
node.resetAttributes(['color', 'radius']);  // Reset specific only
```

## Style Rules

Rules automatically style nodes/edges based on conditions. They re-apply when data changes.

### Basic Rule

```typescript
// Style all nodes
ogma.styles.addNodeRule({
  color: 'blue',
  radius: 8
});

// Style all edges
ogma.styles.addEdgeRule({
  color: 'gray',
  width: 2
});
```

### Data-Driven Rules

```typescript
// Use functions to compute attributes from data
ogma.styles.addNodeRule({
  color: node => node.getData('type') === 'important' ? 'red' : 'gray',
  radius: node => 5 + node.getDegree(),
  text: { content: node => node.getData('name') }
});

ogma.styles.addEdgeRule({
  width: edge => Math.log(edge.getData('weight') + 1),
  color: edge => edge.getData('type') === 'strong' ? '#000' : '#ccc'
});
```

### Selective Rules (with Selectors)

```typescript
// Only apply to nodes matching condition
ogma.styles.addNodeRule(
  node => node.getData('category') === 'person',  // Selector
  {
    color: 'purple',
    icon: { content: '\uf007', font: 'FontAwesome' }
  }
);

// Only style "important" edges
ogma.styles.addEdgeRule(
  edge => edge.getData('priority') > 5,
  { color: 'red', width: 3 }
);
```

### Combined Node/Edge Rule

```typescript
ogma.styles.addRule({
  nodeSelector: node => node.getData('type') === 'server',
  nodeAttributes: {
    color: 'green',
    shape: 'square'
  },
  edgeSelector: edge => edge.getData('encrypted') === true,
  edgeAttributes: {
    color: 'blue',
    shape: 'dashed'
  }
});
```

## Built-in Rule Helpers

```typescript
// Map values to attributes
ogma.styles.addNodeRule({
  color: ogma.rules.map({
    field: 'category',
    values: {
      person: 'blue',
      company: 'green',
      product: 'orange'
    },
    fallback: 'gray'
  })
});

// Slice continuous values into discrete buckets
ogma.styles.addNodeRule({
  radius: ogma.rules.slices({
    field: 'revenue',
    values: { min: 5, max: 20, nbSlices: 5 }  // 5 buckets from 5 to 20
  }),
  color: ogma.rules.slices({
    field: 'score',
    values: ['#fee5d9', '#fcae91', '#fb6a4a', '#cb181d'],  // Color scale
    reverse: false
  })
});
```

## Managing Rules

```typescript
// Add rule and get reference
const rule = ogma.styles.addNodeRule({ color: 'red' });

// Refresh when external data changes
rule.refresh();

// Change priority (lower index = applied first)
rule.setIndex(0);

// Remove rule
rule.destroy();

// Get all rules
const allRules = ogma.styles.getRuleList();

// Clear all rules
ogma.styles.getRuleList().forEach(r => r.destroy());
```

## Classes (Highlight States)

Classes apply temporary styling for states like selection, hover, or custom highlights.

### Built-in Classes

```typescript
// Customize selection appearance
ogma.styles.setSelectedNodeAttributes({
  outerStroke: { color: 'yellow', width: 3 }
});

ogma.styles.setSelectedEdgeAttributes({
  color: 'yellow',
  width: 3
});

// Customize hover appearance
ogma.styles.setHoveredNodeAttributes({
  outerStroke: { color: 'cyan', width: 2 }
});

// Disable selection styling
ogma.styles.setSelectedNodeAttributes(null);
```

### Custom Classes

```typescript
// Create a custom class
const highlightClass = ogma.styles.createClass({
  name: 'highlight',
  nodeAttributes: {
    outerStroke: { color: 'orange', width: 4 },
    halo: { color: 'orange', width: 10 }
  },
  edgeAttributes: {
    color: 'orange',
    width: 3
  }
});

// Apply class to elements
node.addClass('highlight');
nodes.addClass('highlight');
edge.addClass('highlight');

// Remove class
node.removeClass('highlight');

// Check if has class
if (node.hasClass('highlight')) { /* ... */ }

// Clear all elements with this class
highlightClass.clearNodes();
highlightClass.clearEdges();

// Get elements with class
highlightClass.getNodes();  // NodeList
highlightClass.getEdges();  // EdgeList
```

## Batch Styling

```typescript
// Style multiple nodes at once
ogma.getNodes().setAttributes({ color: 'blue' });

// Different styles for each node
const nodes = ogma.getNodes(['n1', 'n2', 'n3']);
nodes.setAttributes([
  { color: 'red' },
  { color: 'green' },
  { color: 'blue' }
]);
```

## Themes

Themes let you replace all default visual values of Ogma with your own styling. Instead of applying rules to override defaults, a theme changes the baseline appearance for nodes, edges, selection, hover, and other visual elements.

```typescript
// Use a pre-built theme
import { midsummerNight } from '@linkurious/ogma-styles';
ogma.styles.setTheme(midsummerNight);

// Or create a custom theme
ogma.styles.setTheme({
  nodeAttributes: {
    color: '#4a90d9',
    radius: 8,
    text: { color: '#333', size: 12 }
  },
  edgeAttributes: {
    color: '#999',
    width: 1.5,
    shape: 'arrow'
  },
  selectedNodeAttributes: {
    outerStroke: { color: '#ff6600', width: 3 }
  },
  selectedEdgeAttributes: {
    color: '#ff6600',
    width: 3
  },
  hoveredNodeAttributes: {
    outerStroke: { color: '#00aaff', width: 2 }
  }
});

// Reset to default theme
ogma.styles.setTheme(null);
```

Themes are the recommended way to establish consistent branding across your visualization. Rules and classes then build on top of the theme for dynamic and state-based styling.

## React Integration

```tsx
import { Ogma, NodeStyle, EdgeStyle } from '@linkurious/ogma-react';

function App() {
  return (
    <Ogma graph={graph}>
      <NodeStyle
        attributes={{ color: 'blue', radius: 10 }}
      />
      <NodeStyle
        selector={node => node.getData('important')}
        attributes={{ color: 'red' }}
      />
      <EdgeStyle
        attributes={{ color: 'gray' }}
      />
    </Ogma>
  );
}
```

## Performance Tips

1. **Prefer constants over functions** in rules for large graphs
2. **Use batch methods**: `nodeList.setAttributes()` not `nodeList.forEach(n => n.setAttributes())`
3. **Limit rule count**: Combine related styling into single rules
4. **Specify dependencies**: For advanced optimization, declare which data fields a rule depends on
