# Tools API

Interactive tools for user interactions: branding, tooltips, snapping, and selection modes.

## Overview

The `ogma.tools` API provides specialized tools for enhancing user interactions with the graph:
- **Brand**: Display custom branding elements in the visualization
- **Tooltip**: Show contextual information on hover, click, or double-click
- **Snapping**: Snap nodes to guidelines while dragging for precise alignment
- **Lasso**: Select nodes and edges by drawing a freeform lasso
- **Rectangle Select**: Select nodes and edges by drawing a rectangle

All tools can be enabled/disabled programmatically and provide customization options.

## Brand

Display HTML content in a corner of the canvas, typically for logos or attribution.

### Setting a Brand

```typescript
// Display brand in bottom-right (default position)
ogma.tools.brand.set('<a href="http://company.com">Powered by MyCompany</a>', {
  position: 'bottom-right',
  horizontalMargin: 10,
  verticalMargin: 10,
  className: 'custom-brand'
});
```

### Brand Options

```typescript
interface BrandOptions {
  position: 'top-left' | 'top-right' | 'bottom-left' | 'bottom-right';
  horizontalMargin?: number;  // Space from top/bottom edge (px)
  verticalMargin?: number;    // Space from left/right edge (px)
  className?: string;         // Custom CSS class
}
```

### Removing the Brand

```typescript
ogma.tools.brand.remove();
```

### Common Use Cases

```typescript
// Logo with image
ogma.tools.brand.set('<img src="logo.png" width="100">', {
  position: 'top-left',
  horizontalMargin: 20,
  verticalMargin: 20
});

// Attribution link
ogma.tools.brand.set(
  '<a href="https://company.com" target="_blank">Data by Company</a>',
  { position: 'bottom-right' }
);
```

**Note**: If the brand appears in the wrong position, ensure the Ogma container has `position: relative` in CSS.

## Tooltip

Display contextual information when hovering or clicking on nodes, edges, or background.

### Node Tooltips

```typescript
// On hover
ogma.tools.tooltip.onNodeHover(
  node => {
    const name = node.getData('name');
    const degree = node.getDegree();
    return `
      <div>
        <strong>${name}</strong>
        <p>Connections: ${degree}</p>
      </div>
    `;
  },
  {
    position: 'top',
    delay: 300,
    className: 'custom-tooltip'
  }
);

// On click
ogma.tools.tooltip.onNodeClick(node => {
  return `<div>Clicked: ${node.getId()}</div>`;
});

// On right-click
ogma.tools.tooltip.onNodeRightClick(node => {
  return `<div>Context menu for ${node.getId()}</div>`;
});

// On double-click
ogma.tools.tooltip.onNodeDoubleClick(node => {
  return `<div>Double-clicked: ${node.getId()}</div>`;
});
```

### Edge Tooltips

```typescript
// On hover
ogma.tools.tooltip.onEdgeHover(edge => {
  const source = edge.getSource().getData('name');
  const target = edge.getTarget().getData('name');
  const relationship = edge.getData('relationship');

  return `
    <div>
      ${source} → ${target}
      <br>Type: ${relationship}
    </div>
  `;
});

// On click, right-click, double-click
ogma.tools.tooltip.onEdgeClick(edge => /* ... */);
ogma.tools.tooltip.onEdgeRightClick(edge => /* ... */);
ogma.tools.tooltip.onEdgeDoubleClick(edge => /* ... */);
```

### Background Tooltips

```typescript
// On background click (e.g., for context menus)
ogma.tools.tooltip.onBackgroundClick(() => {
  return '<div>Add new node</div>';
});

ogma.tools.tooltip.onBackgroundRightClick(() => {
  return '<div>Background context menu</div>';
});

ogma.tools.tooltip.onBackgroundDoubleClick(() => {
  return '<div>Reset view</div>';
});
```

### Tooltip Options

```typescript
interface TooltipOptions {
  position?: 'top' | 'bottom' | 'left' | 'right' | 'cssDefined';
  autoAdjust?: boolean;  // Auto-position at screen edges (default: true)
  delay?: number;        // Delay before showing (ms, default: 0)
  className?: string;    // Custom CSS class
}
```

### Async Tooltips

Tooltip handlers can return Promises for data fetching:

```typescript
ogma.tools.tooltip.onNodeHover(async node => {
  const userId = node.getData('userId');
  const userData = await fetch(`/api/users/${userId}`).then(r => r.json());

  return `
    <div>
      <h3>${userData.name}</h3>
      <p>${userData.email}</p>
    </div>
  `;
});
```

### Tooltip Control

```typescript
// Check if tooltip is visible
if (ogma.tools.tooltip.isShown()) {
  // Hide it
  ogma.tools.tooltip.hide();
}

// Refresh current tooltip
ogma.tools.tooltip.refresh();
```

### Complete Example

```typescript
ogma.tools.tooltip.onNodeHover(
  node => {
    const icon = node.getAttribute('icon.content');
    const text = node.getAttribute('text.content');
    const funding = node.getData('funding_total');
    const status = node.getData('status');
    const degree = node.getDegree();

    let html = `
      <div class="arrow"></div>
      <div class="header">
        <span class="icon">${icon}</span>${text}
      </div>
      <div class="body">
        <table>`;

    if (funding) html += `<tr><th>Funding</th><td>$${funding}</td></tr>`;
    if (status) html += `<tr><th>Status</th><td>${status}</td></tr>`;

    html += `
        </table>
      </div>
      <div class="footer">Connections: ${degree}</div>
    `;

    return html;
  },
  { className: 'custom-tooltip', position: 'top' }
);
```

## Snapping

Snap nodes to alignment guidelines while dragging for precise positioning.

### Enabling Snapping

```typescript
// Enable with defaults
ogma.tools.snapping.enable();

// Enable with custom options
ogma.tools.snapping.enable({
  enabled: true,
  tolerance: 5,
  centerSnapDistance: 240,
  sideSnapDistanceFactor: 3,
  guidelineWidth: 1,
  guidelineColor: 'red',
  preferredDistance: {
    enabled: true,
    ratio: 1.13,
    tolerance: 10,
    lineWidth: 1,
    lineColor: '#00C3FF'
  },
  neighbours: {
    enabled: true,
    lineWidth: 1,
    lineColor: '#00C3FF',
    tolerance: 3,
    offset: 5
  }
});
```

### Snapping Options

```typescript
interface SnappingOptions {
  enabled?: boolean;               // Enable snapping (default: false)
  tolerance?: number;              // Snap distance in pixels (default: 5)
  centerSnapDistance?: number;     // Max distance for center alignment (default: 240)
  sideSnapDistanceFactor?: number; // Factor of node diameter for side alignment (default: 3)
  guidelineWidth?: number;         // Width of guidelines (default: 1)
  guidelineColor?: Color;          // Color of axis-aligned guidelines (default: 'red')

  // Preferred distance snapping (pairwise node distance)
  preferredDistance?: {
    enabled?: boolean;             // Enable mode (default: true)
    ratio?: number;                // Distance ratio d = (Ra + Rb) * ratio (default: 1.13)
    tolerance?: number;            // Snap distance (default: 10)
    lineWidth?: number;            // Guideline width (default: 1)
    lineColor?: Color;             // Guideline color (default: '#00C3FF')
  };

  // Equal distribution snapping (middle of distance between nodes)
  neighbours?: {
    enabled?: boolean;             // Enable mode (default: true)
    lineWidth?: number;            // Guideline width (default: 1)
    lineColor?: Color;             // Guideline color (default: '#00C3FF')
    tolerance?: number;            // Snap distance (default: 3)
    offset?: number;               // Distance from aligned nodes (default: 5)
  };
}
```

### Snapping Modes

**Axis alignment**: Aligns node centers or sides horizontally/vertically with other nodes.

**Preferred distance**: Snaps nodes to maintain consistent spacing based on their sizes.

**Neighbour distribution**: Snaps nodes to equal distances between neighbours.

### Disabling Snapping

```typescript
ogma.tools.snapping.disable();

// Check if enabled
if (ogma.tools.snapping.enabled()) {
  console.log('Snapping is active');
}
```

### Interactive Example

```typescript
// Toggle snapping with checkbox
document.getElementById('snapping-checkbox').addEventListener('change', e => {
  if (e.target.checked) {
    ogma.tools.snapping.enable({
      neighbours: { enabled: true },
      preferredDistance: { enabled: true }
    });
  } else {
    ogma.tools.snapping.disable();
  }
});
```

## Lasso Selection

Select nodes and edges by drawing a freeform lasso shape.

### Enabling Lasso

```typescript
// Enable with defaults (adds to selection on completion)
ogma.tools.lasso.enable();

// Enable with custom options
ogma.tools.lasso.enable({
  strokeColor: '#00C3FF',
  strokeWidth: 1,
  fillColor: 'rgba(0, 195, 255, 0.1)',
  cursorStyle: 'crosshair',
  bothExtremities: false,
  callback: evt => {
    // Custom behavior on lasso completion
    console.log('Selected nodes:', evt.nodes.getId());
    console.log('Selected edges:', evt.edges.getId());

    evt.nodes.setSelected(true);
    evt.edges.setSelected(true);
  }
});
```

### Lasso Options

```typescript
interface LassoOptions {
  strokeColor?: Color;           // Lasso outline color (default: '#00C3FF')
  strokeWidth?: number;          // Lasso outline width (default: 1)
  fillColor?: Color;             // Lasso fill color (default: 'rgba(0,195,255,0.1)')
  cursorStyle?: CursorStyle;     // CSS cursor style (default: 'cell')
  bothExtremities?: boolean;     // Both edge endpoints must be inside (default: false)
  callback?: (payload: {
    nodes: NodeList;
    edges: EdgeList;
    points: Point[];
  }) => void;
}
```

### Callback Payload

```typescript
{
  nodes: NodeList;    // Nodes inside the lasso
  edges: EdgeList;    // Edges inside the lasso
  points: Point[];    // Lasso path coordinates
}
```

### Disabling Lasso

```typescript
ogma.tools.lasso.disable();

// Check if enabled
if (ogma.tools.lasso.enabled()) {
  console.log('Lasso is active');
}
```

### Events

```typescript
ogma.events.on('lassoEnabled', () => {
  console.log('Lasso tool activated');
});

ogma.events.on('lassoDisabled', () => {
  console.log('Lasso tool deactivated');
});
```

### Conditional Activation

```typescript
// Enable lasso only while Ctrl is pressed
ogma.events.on('dragStart', () => {
  if (ogma.keyboard.isKeyPressed('ctrl')) {
    ogma.clearSelection();
    ogma.tools.lasso.enable({
      bothExtremities: true,
      callback: evt => {
        evt.nodes.setSelected(true);
        evt.edges.setSelected(true);
      }
    });
  }
});
```

### UI Integration

```typescript
const lassoButton = document.getElementById('lasso-button');

lassoButton.addEventListener('click', () => {
  if (ogma.tools.lasso.enabled()) {
    ogma.tools.lasso.disable();
  } else {
    ogma.tools.lasso.enable();
  }
});

// Visual feedback
ogma.events
  .on('lassoEnabled', () => lassoButton.classList.add('active'))
  .on('lassoDisabled', () => lassoButton.classList.remove('active'));
```

## Rectangle Selection

Select nodes and edges by drawing a rectangular area.

### Enabling Rectangle Select

```typescript
// Enable with defaults (adds to selection on completion)
ogma.tools.rectangleSelect.enable();

// Enable with custom options
ogma.tools.rectangleSelect.enable({
  strokeColor: 'blue',
  strokeWidth: 2,
  fillColor: 'rgba(0, 0, 255, 0.1)',
  cursorStyle: 'crosshair',
  bothExtremities: false,
  callback: evt => {
    // Custom behavior: replace selection instead of adding
    ogma.clearSelection();
    evt.nodes.setSelected(true);
    evt.edges.setSelected(true);
  }
});
```

### Rectangle Select Options

```typescript
interface RectangleSelectOptions {
  strokeColor?: Color;           // Rectangle outline color (default: '#00C3FF')
  strokeWidth?: number;          // Rectangle outline width (default: 1)
  fillColor?: Color | null;      // Rectangle fill color (default: 'rgba(0,195,255,0.1)')
  cursorStyle?: CursorStyle;     // CSS cursor style (default: 'cell')
  bothExtremities?: boolean;     // Both edge endpoints must be inside (default: false)
  callback?: (payload: {
    nodes: NodeList;
    edges: EdgeList;
    rectangle: SimpleBoundingBox;
  }) => void;
}
```

### Callback Payload

```typescript
{
  nodes: NodeList;              // Nodes inside the rectangle
  edges: EdgeList;              // Edges inside the rectangle
  rectangle: SimpleBoundingBox; // { x, y, width, height }
}
```

### Disabling Rectangle Select

```typescript
ogma.tools.rectangleSelect.disable();

// Check if enabled
if (ogma.tools.rectangleSelect.enabled()) {
  console.log('Rectangle select is active');
}
```

### Events

```typescript
ogma.events.on('rectangleSelectEnabled', () => {
  console.log('Rectangle select activated');
});

ogma.events.on('rectangleSelectDisabled', () => {
  console.log('Rectangle select deactivated');
});
```

### Selection Modes

```typescript
// Additive mode (default): add to existing selection
ogma.tools.rectangleSelect.enable({
  callback: evt => {
    evt.nodes.setSelected(true);
    evt.edges.setSelected(true);
  }
});

// Replace mode: clear selection first
ogma.tools.rectangleSelect.enable({
  callback: evt => {
    ogma.clearSelection();
    evt.nodes.setSelected(true);
    evt.edges.setSelected(true);
  }
});

// Nodes only: ignore edges
ogma.tools.rectangleSelect.enable({
  callback: evt => {
    evt.nodes.setSelected(true);
  }
});
```

### UI Integration

```typescript
const rectButton = document.getElementById('rect-select-button');

rectButton.addEventListener('click', () => {
  if (ogma.tools.rectangleSelect.enabled()) {
    ogma.tools.rectangleSelect.disable();
  } else {
    ogma.tools.rectangleSelect.enable({
      strokeColor: 'blue',
      callback: evt => {
        ogma.clearSelection();
        evt.nodes.setSelected(true);
      }
    });
  }
});

// Visual feedback
ogma.events
  .on('rectangleSelectEnabled', () => rectButton.classList.add('active'))
  .on('rectangleSelectDisabled', () => rectButton.classList.remove('active'));
```

## Edge Extremity Handling

Both lasso and rectangle select support the `bothExtremities` option:

```typescript
// Default: include edges if ANY endpoint is inside
ogma.tools.lasso.enable({
  bothExtremities: false,
  callback: evt => {
    // evt.edges includes edges with at least one endpoint inside
  }
});

// Strict: include edges only if BOTH endpoints are inside
ogma.tools.rectangleSelect.enable({
  bothExtremities: true,
  callback: evt => {
    // evt.edges includes only fully enclosed edges
  }
});
```

## Performance Tips

1. **Tooltip delays**: Use `delay` option to avoid frequent tooltip updates during fast mouse movement
2. **Async tooltips**: Show loading state immediately, then update with fetched data
3. **Selection callbacks**: Batch operations using NodeList/EdgeList methods instead of iterating
4. **Snapping**: Disable when not needed to improve dragging performance on large graphs
5. **Event listeners**: Remove unnecessary event listeners when tools are disabled

## Common Patterns

### Toggle Tools with Keyboard Shortcuts

```typescript
ogma.events.on('keydown', evt => {
  if (evt.key === 'l') {
    // Toggle lasso
    if (ogma.tools.lasso.enabled()) {
      ogma.tools.lasso.disable();
    } else {
      ogma.tools.lasso.enable();
    }
  } else if (evt.key === 'r') {
    // Toggle rectangle select
    if (ogma.tools.rectangleSelect.enabled()) {
      ogma.tools.rectangleSelect.disable();
    } else {
      ogma.tools.rectangleSelect.enable();
    }
  }
});
```

### Mutually Exclusive Tools

```typescript
function activateTool(tool: 'lasso' | 'rectangle' | 'none') {
  // Disable all tools first
  ogma.tools.lasso.disable();
  ogma.tools.rectangleSelect.disable();

  // Enable the requested tool
  if (tool === 'lasso') {
    ogma.tools.lasso.enable();
  } else if (tool === 'rectangle') {
    ogma.tools.rectangleSelect.enable();
  }
}
```

### Rich Tooltips with Templates

```typescript
function createTooltipHTML(title: string, data: Record<string, any>) {
  const rows = Object.entries(data)
    .map(([key, value]) => `
      <tr>
        <th>${key}</th>
        <td>${value}</td>
      </tr>
    `)
    .join('');

  return `
    <div class="tooltip-container">
      <div class="tooltip-header">${title}</div>
      <div class="tooltip-body">
        <table>${rows}</table>
      </div>
    </div>
  `;
}

ogma.tools.tooltip.onNodeHover(node => {
  return createTooltipHTML(
    node.getData('name'),
    {
      Type: node.getData('type'),
      Status: node.getData('status'),
      Connections: node.getDegree()
    }
  );
});
```
