# Transformations

Graph transformations in Ogma: node grouping, edge grouping, and visual grouping.

## Overview

Transformations modify how the graph is displayed without changing the underlying data. They can:
- Group nodes into meta-nodes
- Group parallel edges into single edges
- Show/hide group contents (visual grouping)

All transformations return a handle for enabling, disabling, and destroying.

## Node Grouping

Combine multiple nodes into a single meta-node.

### Basic Grouping by Property

```typescript
const grouping = ogma.transformations.addNodeGrouping({
  // Group nodes by their 'country' property
  groupIdFunction: node => node.getData('country'),

  // Generate the meta-node for each group
  nodeGenerator: (nodes, groupId) => ({
    id: `group-${groupId}`,
    data: {
      label: groupId,
      count: nodes.size
    },
    attributes: {
      text: { content: `${groupId} (${nodes.size})` },
      radius: 10 + nodes.size * 2,
      color: 'purple'
    }
  })
});

// Wait for grouping to complete
await grouping.whenApplied();
```

### Selective Grouping

Only group nodes matching a condition:

```typescript
const grouping = ogma.transformations.addNodeGrouping({
  // Only group nodes where type is 'user'
  selector: node => node.getData('type') === 'user',

  // Group by department
  groupIdFunction: node => node.getData('department'),

  nodeGenerator: (nodes, groupId) => ({
    id: `dept-${groupId}`,
    attributes: {
      text: `${groupId} Department`,
      radius: Math.max(10, nodes.size * 3)
    }
  })
});
```

### Controlling Grouped Edges

When nodes are grouped, their edges are combined. Control this behavior:

```typescript
const grouping = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => node.getData('cluster'),
  nodeGenerator: (nodes, groupId) => ({
    id: `cluster-${groupId}`,
    attributes: { radius: 20 }
  }),

  // Group edges between meta-nodes
  groupEdges: true,

  // Keep edge direction separate
  separateEdgesByDirection: true,

  // Generate meta-edge attributes
  edgeGenerator: (edges, groupId, source, target) => ({
    id: `edge-${source.getId()}-${target.getId()}`,
    data: { count: edges.size },
    attributes: {
      width: Math.log(edges.size + 1) * 2,
      text: `${edges.size} connections`
    }
  })
});
```

## Edge Grouping

Combine parallel edges (multiple edges between same node pair) into single edges.

### Basic Edge Grouping

```typescript
const edgeGrouping = ogma.transformations.addEdgeGrouping();

await edgeGrouping.whenApplied();
```

### Custom Edge Grouping

```typescript
const edgeGrouping = ogma.transformations.addEdgeGrouping({
  // Only group edges of type 'transaction'
  selector: edge => edge.getData('type') === 'transaction',

  // Generate meta-edge
  generator: (edges, groupId, source, target) => {
    const totalAmount = edges.getData('amount').reduce((sum, a) => sum + a, 0);
    return {
      id: `grouped-${source.getId()}-${target.getId()}`,
      data: {
        totalAmount,
        count: edges.size
      },
      attributes: {
        width: Math.log(totalAmount + 1),
        text: `$${totalAmount.toLocaleString()}`
      }
    };
  }
});
```

### Group by Edge Property

```typescript
const edgeGrouping = ogma.transformations.addEdgeGrouping({
  // Group parallel edges by their 'type' property
  groupIdFunction: edge => edge.getData('type'),

  generator: (edges, groupId) => ({
    id: `${groupId}-edge`,
    attributes: {
      color: groupId === 'friend' ? 'green' : 'gray',
      text: `${groupId} (${edges.size})`
    }
  })
});
```

## Visual Grouping

Show group contents as nested nodes inside a container (compound nodes).

### Basic Visual Grouping

```typescript
const grouping = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => node.getData('cluster'),

  // Show contents of groups (key difference from regular grouping)
  showContents: true,

  // Padding inside group container
  padding: 20,

  nodeGenerator: (nodes, groupId) => ({
    id: `cluster-${groupId}`,
    attributes: {
      color: 'rgba(100, 100, 200, 0.2)',  // Semi-transparent container
      text: { content: groupId, position: 'top' }
    }
  })
});
```

### Dynamic Open/Close Groups

```typescript
const grouping = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => node.getData('category'),

  // Function to determine if group should be open
  showContents: metaNode => metaNode.getData('isOpen') === true,

  nodeGenerator: (nodes, groupId) => ({
    id: `group-${groupId}`,
    data: { isOpen: false },  // Start closed
    attributes: { text: groupId }
  })
});

// Toggle group open/close on double-click
ogma.events.on('doubleclick', async (event) => {
  const target = event.target;
  if (target?.isNode && target.getId().toString().startsWith('group-')) {
    const isOpen = target.getData('isOpen');
    target.setData('isOpen', !isOpen);
    await grouping.refresh();
  }
});
```

### Layout Group Contents

```typescript
const grouping = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => node.getData('department'),
  showContents: true,
  padding: 30,

  // Layout contents when group is created/updated
  onGroupUpdate: async (metaNode, subNodes, isOpen) => {
    if (isOpen && subNodes.size > 1) {
      // Apply force layout to group contents
      await ogma.layouts.force({
        nodes: subNodes,
        charge: 5,
        steps: 100
      });
    }
  },

  nodeGenerator: (nodes, groupId) => ({
    id: `dept-${groupId}`,
    data: { isOpen: true },
    attributes: {
      color: 'rgba(200, 200, 200, 0.3)',
      text: { content: groupId, position: 'top' }
    }
  })
});

// Layout meta-nodes after grouping
await grouping.whenApplied();
await ogma.layouts.force();
```

## Transformation Control

### Enable/Disable

```typescript
const grouping = ogma.transformations.addNodeGrouping({ /* ... */ });

// Disable (ungroup)
await grouping.disable();

// Enable (re-apply grouping)
await grouping.enable();

// Toggle
await grouping.toggle();

// Check state
if (grouping.isEnabled()) { /* ... */ }
```

### Refresh

Re-apply transformation when data changes:

```typescript
// Data changed externally
node.setData('category', 'newCategory');

// Refresh to re-group
await grouping.refresh();
```

### Destroy

Remove transformation completely:

```typescript
await grouping.destroy();
```

### Animation

```typescript
const grouping = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => node.getData('cluster'),
  duration: 500,  // Animate grouping over 500ms
  easing: 'quadraticOut',
  /* ... */
});

// Animate disable/enable
await grouping.disable(300);  // 300ms animation
await grouping.enable(300);
```

## Accessing Grouped Elements

```typescript
// Get the meta-node for a grouped node
const originalNode = ogma.getNode('user1');
const metaNode = originalNode.getMetaNode();

// Get original nodes inside a meta-node
const metaNode = ogma.getNode('group-sales');
const contents = metaNode.getSubNodes();  // NodeList of grouped nodes

// Check if node is a meta-node
if (node.isVirtual()) {
  console.log('This is a meta-node');
}

// Check if node is grouped
if (node.getMetaNode() !== null) {
  console.log('This node is inside a group');
}
```

## Multiple Transformations

Transformations can be chained:

```typescript
// First: group by country
const countryGrouping = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => node.getData('country'),
  /* ... */
});

// Then: group by region (groups the country groups)
const regionGrouping = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => node.getData('region'),
  /* ... */
});

// Order matters - later transformations apply to results of earlier ones
```

## Common Patterns

### Expand/Collapse on Click

```typescript
const grouping = ogma.transformations.addNodeGrouping({
  groupIdFunction: node => node.getData('cluster'),
  showContents: metaNode => metaNode.getData('expanded'),
  nodeGenerator: (nodes, groupId) => ({
    id: `cluster-${groupId}`,
    data: { expanded: false }
  })
});

ogma.events.on('click', async (event) => {
  const target = event.target;
  if (target?.isNode && target.isVirtual()) {
    target.setData('expanded', !target.getData('expanded'));
    await grouping.refresh();
  }
});
```

### Hierarchical Grouping

```typescript
// Level 1: Group by department
const deptGrouping = ogma.transformations.addNodeGrouping({
  selector: node => node.getData('type') === 'employee',
  groupIdFunction: node => node.getData('department'),
  showContents: true,
  nodeGenerator: (nodes, groupId) => ({
    id: `dept-${groupId}`,
    data: { level: 1 }
  })
});

// Level 2: Group departments by division
const divisionGrouping = ogma.transformations.addNodeGrouping({
  selector: node => node.getData('level') === 1,
  groupIdFunction: node => node.getData('division'),
  showContents: true,
  nodeGenerator: (nodes, groupId) => ({
    id: `div-${groupId}`,
    data: { level: 2 }
  })
});
```

## React Integration

```tsx
import { Ogma, Transformation } from '@linkurious/ogma-react';

function App() {
  return (
    <Ogma graph={graph}>
      <Transformation
        type="nodeGrouping"
        groupIdFunction={node => node.getData('category')}
        nodeGenerator={(nodes, groupId) => ({
          id: `group-${groupId}`,
          attributes: { text: groupId }
        })}
      />
    </Ogma>
  );
}
```

## Performance Tips

1. **Use selectors** to limit which nodes are considered for grouping
2. **Batch data changes** before calling `refresh()`
3. **Use `duration: 0`** for instant grouping on large graphs
4. **Avoid deeply nested** visual grouping (performance degrades with depth)
