# Ogma

## `ogma.constructor([parameters])`

**Arguments**
  * `parameters`(optional) `object`
      * `container`(optional) `string|HTMLElement` HTML element to use as a container.
If a string is specified, the element will be lookup with `document.getElementById()`.
      * `dimensions`(optional) `Size`[default: { width: 300, height: 300 }] Dimensions of the canvas. If unspecified, the container's dimensions will be used.
      * `graph`(optional) `RawGraph` Graph to initialize Ogma with
      * `imgCrossOrigin`(optional) `CrossOriginValue`[default: "anonymous"] Indicates the value of the `crossOrigin` field for DOM images.
This is an alias for `ogma.setOptions({imgCrossOrigin: value})`
      * `options`(optional) `Options` Settings for all the modules.
      * `renderer`(optional) `"webgl"|"canvas"|"svg"|null`[default: "webgl"] Rendering type. If WebGL is selected and not available, Ogma
fallback on Canvas. If no renderer is available (e.g in Node.js), Ogma will fallback on headless mode (`null`).
This field is an alias for `ogma.setOptions({renderer: value})`

## `ogma.addEdge(edge[, options])`

Add the specified edge to the graph

**Arguments**
  * `edge` `RawEdge` Edge to add.
  * `options`(optional) `AddItemOptions`
      * `ignoreInvalid`(optional) `boolean`[default: false] If true, the method quietly
skip the edges whose extremities are not in the visualisation.

**Returns**
  *  `Edge` Edge that has just been added.

## `ogma.addEdges(edges[, options])`

Add the specified edges to the graph

**Arguments**
  * `edges` `Array<RawEdge>`
  * `options`(optional) `object`
      * `batchSize`(optional) `number` If specified, the graph will be
                                    imported progressively (`batchSize`
                                    nodes/edges at a time). It will
                                    effectively increase the total loading
                                    time, but the construction of the graph
                                    will be shown and the thread will not
                                    be frozen.
      * `ignoreInvalid`(optional) `boolean`[default: false] If true, the method quietly
skip the edges whose extremities are not in the visualisation.

**Returns**
  *  `Promise<EdgeList>` Edges added to the graph.

## `ogma.addGraph(graph[, options])`

Add the specified nodes and edges to the graph. If your visualisation is empty, it will also
automatically center the view on the graph.

**Arguments**
  * `graph` `RawGraph`
  * `options`(optional) `object`
      * `batchSize`(optional) `number` If specified, the graph will be
imported progressively (`batchSize` nodes/edges at a time). It will
effectively increase the total loading time, but the construction of the
graph will be shown and the thread will not be frozen.
      * `ignoreInvalid`(optional) `boolean`[default: false] If true, the method quietly
skip the edges whose extremities are not in the visualisation.

**Returns**
  *  `Promise<{nodes: NodeList, edges: EdgeList}>`

## `ogma.addNode(node[, options])`

Add the specified node to the graph

**Arguments**
  * `node` `RawNode` Node to add.
  * `options`(optional) `AddItemOptions` Unused for now.

**Returns**
  *  `Node` Node that has just been added.

## `ogma.addNodes(nodes[, options])`

Add the specified nodes to the graph. Ignores nodes that have the same id
as a node in the graph.

**Arguments**
  * `nodes` `Array<RawNode>`
  * `options`(optional) `object`
      * `batchSize`(optional) `number` If specified, the graph will be
                                    imported progressively (`batchSize`
                                    nodes/edges at a time). It will
                                    effectively increase the total loading
                                    time, but the construction of the graph
                                    will be shown and the thread will not
                                    be frozen.

**Returns**
  *  `Promise<NodeList>` Nodes added to the graph.

## `ogma.clearGraph()`

Removes all the nodes and edges from the graph.

**Returns**
  *  `void`

## `ogma.clearSelection()`

Clear the selection.

## `ogma.createEdgeList()`

Returns a new empty EdgeList.

**Returns**
  *  `EdgeList`

## `ogma.createNodeList()`

Returns a new empty NodeList.

**Returns**
  *  `NodeList`

## `ogma.destroy()`

Release the memory used and removes all connections between Ogma and the
DOM. After this method is called, Ogma's container will not contain any DOM
element  created by Ogma. Global DOM elements such as `window` and
`document`'s event listeners added by Ogma will be removed. All
`setTimeout` created by Ogma will be cleared.

**Returns**
  *  `void`

## `ogma.getConnectedComponentByNode(node[, options])`

**Arguments**
  * `node` `Node|NodeId`
  * `options`(optional) `object`
      * `filter`(optional) `object`[default: &#039;visible&#039;]
      * `returnIds`(optional) `boolean`[default: false] Return node ids instead of Nodes

**Returns**
  *  `NodeList`

## `ogma.getConnectedComponents([options])`

Returns weakly connected components of the graph.

**Arguments**
  * `options`(optional) `object`
      * `filter`(optional) `object`[default: &#039;visible&#039;]
      * `returnIds`(optional) `boolean`[default: false] Return node ids instead of Nodes

**Returns**
  *  `Array<NodeList>`

## `ogma.getContainer()`

Returns the DOM element used by this Ogma instance.

**Returns**
  *  `HTMLElement|null`

## `ogma.getDisabledEdges()`

Returns all edges that are disabled.

**Returns**
  *  `EdgeList`

## `ogma.getDisabledNodes()`

Returns all nodes that are disabled.

**Returns**
  *  `NodeList`

## `ogma.getEdge(edgeId)`

Return the specified edge, or `undefined` if it doesn't exist.

**Arguments**
  * `edgeId` `EdgeId`

**Returns**
  *  `Edge|undefined`

## `ogma.getEdgeFilters()`

Retrieve all edge filters.

Same as
```js
const edgeFilters = ogma.transformations.getEdgeFilters();
```

**Returns**
  *  `Array<Transformation>`

## `ogma.getEdges([selector])`

Return the specified edges.

**Arguments**
  * `selector`(optional) `Array<EdgeId>|Filter|Array<Edge>`[default: "visible"] If it's an
array of ids, returns the edges that match the specified ids.
If it's "visible", return all the visible edges. If it's "raw", returns all
edges except the ones that are the result of a transformation (e.g. grouping).
If it's "all", returns all the edges.

**Returns**
  *  `EdgeList`

## `ogma.getEdgesByClassName(className[, filter])`

Returns the edges that have the specified class.
Same effect as `StyleClass.update`.

**Arguments**
  * `className` `string`
  * `filter`(optional) `Filter` Filter to apply to edges

**Returns**
  *  `EdgeList`

## `ogma.getEnabledEdges()`

Returns all edges that are not disabled (enabled).

**Returns**
  *  `EdgeList`

## `ogma.getEnabledNodes()`

Returns all nodes that are not disabled (enabled).

**Returns**
  *  `NodeList`

## `ogma.getHoveredElement()`

Returns the element that is currently hovered.

**Returns**
  *  `Node|Edge|null`

## `ogma.getNode(nodeId)`

Return the specified node, or `undefined` if it doesn't exist.

**Arguments**
  * `nodeId` `NodeId`

**Returns**
  *  `Node|undefined`

## `ogma.getNodeFilters()`

Retrieve all node filters.

Same as
```js
const nodeFilters = ogma.transformations.getNodeFilters();
```

**Returns**
  *  `Array<Transformation>`

## `ogma.getNodes([selector])`

Return the specified nodes.

**Arguments**
  * `selector`(optional) `Array<NodeId>|Filter|Array<Node>`[default: "visible"] If it's an
      array of ids, returns the nodes that match the specified ids.
If it's "visible", return all the visible nodes. If it's "raw", returns all
nodes except the ones that are the result of a transformation
(e.g. grouping). If it's "all", returns all the nodes.

**Returns**
  *  `NodeList`

## `ogma.getNodesByClassName(className[, filter])`

Returns the nodes that have the specified class.
Same effect as `StyleClass.update`.

**Arguments**
  * `className` `string`
  * `filter`(optional) `Filter` Filter to apply to nodes

**Returns**
  *  `NodeList`

## `ogma.getNonSelectedEdges()`

Returns all edges that are not selected.

**Returns**
  *  `EdgeList`

## `ogma.getNonSelectedNodes()`

Returns all nodes that are not selected.

**Returns**
  *  `NodeList`

## `ogma.getOptions()`

Get options of the Ogma instance.

**Returns**
  *  `Options`

## `ogma.getPointerInformation()`

Returns information on the cursor.

**Returns**
  *  `{x: number, y: number, target: Node|Edge|null}`

## `ogma.getSelectedEdges()`

Returns all edges that are selected.

**Returns**
  *  `EdgeList`

## `ogma.getSelectedNodes()`

Returns all nodes that are selected.

**Returns**
  *  `NodeList`

## `ogma.isDestroyed()`

**Returns**
  *  `boolean`

## `ogma.reloadFonts()`

Indicates that the DOM has finished loading fonts. If you use an external font (like FontAwesome) and the font
is not displayed correctly on your nodes and edges (e.g squares instead of the actual characters), call this
method once you know the font has been loaded.

## `ogma.removeEdge(edge)`

Remove the specified edge from the graph.

**Arguments**
  * `edge` `Edge|EdgeId`

## `ogma.removeEdges(edges)`

Remove the specified edges from the graph

**Arguments**
  * `edges` `EdgeList|Array<Edge|EdgeId>`

**Returns**
  *  `Promise<void>`

## `ogma.removeNode(node)`

Remove the specified node from the graph.

**Arguments**
  * `node` `Node|NodeId`

## `ogma.removeNodes(nodes)`

Remove the specified nodes from the graph

**Arguments**
  * `nodes` `NodeList|Array<Node|NodeId>`

**Returns**
  *  `Promise<void>`

## `ogma.reset()`

Reset Ogma to its initial state. Doing `ogma.reset();` has the same effect as
`ogma.destroy(); ogma = new Ogma(params);`, with `params` being the parameters used the
first time Ogma was instantiated.

**Returns**
  *  `void`

## `ogma.setContainer(elt)`

Set the DOM element used by this Ogma instance. If a string is specified,
the element will be looked up with `document.getElementById()`.
If the argument is `null`, then Ogma is removed from the current container.

**Arguments**
  * `elt` `HTMLElement|string|null`

## `ogma.setGraph(graph[, options])`

Clear the graph, then add the specified nodes and edges to the graph.

**Arguments**
  * `graph` `RawGraph`
  * `options`(optional) `object`
      * `batchSize`(optional) `number` If specified, the graph will be
imported progressively (`batchSize` nodes/edges at a time). It will
effectively increase the total loading time, but the construction of the
graph will be shown and the thread will not be frozen.
      * `ignoreInvalid`(optional) `boolean`[default: false] If true, the method quietly
skip the edges whose extremities are not in the visualisation.

**Returns**
  *  `Promise<{nodes: NodeList, edges: EdgeList}>`

## `ogma.setOptions(options)`

Update the options of Ogma.

**Arguments**
  * `options` `Options`

# Ogma.algorithms

Algorithms namespace of Ogma, everything to calculate metrics and run analysis on the graph.

## `ogma.algorithms.betweenness([options])`

Returns the betweenness score for the given nodes

**Arguments**
  * `options`(optional) `BetweennessOptions`

## `ogma.algorithms.bfs(options)`

Breadth first search.

**Arguments**
  * `options` `TraversalOptions`

## `ogma.algorithms.circlePack(options)`

**Arguments**
  * `options` `object`
      * `duration`(optional) `number`[default: 0] Animation duration
      * `easing`(optional) `Easing`[default: &#039;linear&#039;] Animation easing
      * `margin`(optional) `number`[default: 0] Margin between nodes
      * `nodes` `NodeList` Nodes to pack
      * `origin`(optional) `Point`[default: {x: 0, y: 0}] Origin of the circle pack
      * `sort`(optional) `function(a: Node, b: Node): number | 'asc' | 'desc'` Sorting comparator function or 'asc' or 'desc' to sort by radius in ascending or descending order from the center

**Returns**
  *  `Promise<NodeList>` A promise resolving to the nodes with their new positions.

## `ogma.algorithms.detectCycle([options])`

Returns the first cycle found as a NodeList.

**Arguments**
  * `options`(optional) `object`
      * `edges`(optional) `EdgeList` If omitted, adjacent edges of the
                                    provided nodes are going to be used.
      * `nodes`(optional) `NodeList` If omitted, the whole graph will be taken as input.

**Returns**
  *  `NodeList|null`

## `ogma.algorithms.dfs(options)`

Depth first search.

**Arguments**
  * `options` `TraversalOptions`

## `ogma.algorithms.fishEyeExpand(options)`

**Arguments**
  * `options` `object`
      * `deltaRadius` `number` Radius difference of the expanded node
      * `duration`(optional) `number`[default: 0] Animation duration
      * `easing`(optional) `Easing`[default: &#039;linear&#039;] Animation easing
      * `focusNode` `Node` Node to update the radius
      * `nodes` `NodeList` Nodes to move

**Returns**
  *  `Promise<NodeList>` A promise resolving to the nodes with their new positions.

## `ogma.algorithms.getAllSimpleCycles([options])`

Implements Tarjan's algorithm of finding all simple cycles in the directed graph.

**Arguments**
  * `options`(optional) `object`
      * `edges`(optional) `EdgeList` If omitted, adjacent edges of the
                                    provided nodes are going to be used.
      * `nodes`(optional) `NodeList` If omitted, the whole graph will be taken as input.

**Returns**
  *  `Array<NodeList>`

## `ogma.algorithms.getMinimumEnclosingCircle(nodes)`

Calculates the minimum enclosing circle for the given nodes.
It will throw an error if no nodes are provided.

**Arguments**
  * `nodes` `NodeList`

**Returns**
  *  `{ x: number, y: number, radius: number }` The center and radius of the circle.

## `ogma.algorithms.hasCycle([options])`

Checks whether the given graph has cycles in it.

**Arguments**
  * `options`(optional) `object`
      * `edges`(optional) `EdgeList` If omitted, adjacent edges of the
                                    provided nodes are going to be used.
      * `nodes`(optional) `NodeList` If omitted, the whole graph will be taken as input.

**Returns**
  *  `Boolean`

## `ogma.algorithms.minimumSpanningTree([nodes][, edges][, edgeCostFunction])`

Kruskal's minimum-spanning-tree algorithm. It finds the edge set for the
graph of the least possible edge cost that connects all the nodes of
the graph.

**Arguments**
  * `nodes`(optional) `NodeList` Nodes of the subgraph to analyze. By default uses all the visible nodes.
  * `edges`(optional) `EdgeList` Edges of the subgraph to analyze. By default all the visible edges.
  * `edgeCostFunction`(optional) `function(edge: Edge):number` Function to get the
weight of the edge, for instance it can take it from the data fields.

**Returns**
  *  `Array<{nodes: NodeList, edges: EdgeList}>` List of minimum spanning trees of the graph

## `ogma.algorithms.shortestPath(options)`

Compute the shortest path between the specified source and target nodes.

**Arguments**
  * `options` `object`
      * `directed`(optional) `boolean`[default: false] Indicates if the graph should be considered as directed.
      * `edgeCostFunction`(optional) `function(edge: Edge): number` Function retrieving the cost of an edge. By default, returns 1 for all edges.
      * `edges`(optional) `EdgeList`[default: undefined] Indicates on which elements to perform the algorithm. If not specified, allow all visible edges.
      * `heuristicFunction`(optional) `function(source: Node, target: Node): number` Function retrieving an estimation
of the distance between two nodes. By default no heuristic is used.
      * `nodes`(optional) `NodeList`[default: undefined] Indicates on which elements to perform the algorithm. If not specified, allow all visible nodes.
      * `source` `Node|NodeId`
      * `target` `Node|NodeId`

**Returns**
  *  `Promise<null|{nodes: NodeList, edges: EdgeList}>` Shortest path. `nodes` has exactly one more node than `edges` has edges.
If there is no path, returns `null`.

# Ogma.events

Events API module: listen to events triggered by the Ogma instance to add interactivity to your application.

## `ogma.events.off(listener)`

Remove a listener from all events it was bound to.

**Arguments**
  * `listener` `function`

## `ogma.events.on(eventName, listener)`

Listen to an event and call the listener function.

**Arguments**
  * `eventName` `string|string[]` Can be an event or an array of events
  * `listener` `function`

## `ogma.events.onEdgesClassAdded(className, listener)`

Triggers the specified function when the specified class is added to some edges.

**Arguments**
  * `className` `string`
  * `listener` `function(evt: {edges: EdgeList})`

## `ogma.events.onEdgesClassRemoved(className, listener)`

Triggers the specified function when the specified class is removed from some edges.

**Arguments**
  * `className` `string`
  * `listener` `function(evt: {edges: EdgeList})`

## `ogma.events.onKeyPress(key, listener)`

Triggers the specified function when the specified key is pressed.

**Arguments**
  * `key` `KeyName|KeyCode|Array<KeyName|KeyCode>|string` Key to listen to. Multiple keys can be specified; in
that case the function is triggered when the last key of the list is pressed, only if all the other keys are pressed.
  * `listener` `function(evt: {domEvent: Event})`

## `ogma.events.onNodesClassAdded(className, listener)`

Triggers the specified function when the specified class is added to some nodes.

**Arguments**
  * `className` `string`
  * `listener` `function(evt: {nodes: NodeList})`

## `ogma.events.onNodesClassRemoved(className, listener)`

Triggers the specified function when the specified class is removed from some nodes.

**Arguments**
  * `className` `string`
  * `listener` `function(evt: {nodes: NodeList})`

## `ogma.events.once(eventName, listener)`

Listen to an event only once.

**Arguments**
  * `eventName` `string|string[]` Can be an event or an array of events
  * `listener` `function`

## `ogma.events.onAnimate(listener)` **DEPRECATED**

Deprecated, use `animate` event instead.
Trigger the specified function when the animation is called on nodes or edges.

**Arguments**
  * `listener` `function(evt: {elements: NodeList|EdgeList, duration: number, easing: Easing })`

## `ogma.events.onBeforeEdgesRemoved(listener)` **DEPRECATED**

Deprecated, use `beforeRemoveEdges` event instead.
Triggers right before the edges are removed, but they are still in the
graph and their data is accessible.

**Arguments**
  * `listener` `function(evt: {edges: EdgeList})`

## `ogma.events.onBeforeNodesRemoved(listener)` **DEPRECATED**

Deprecated, use `beforeRemoveNodes` event instead.
Triggers right before the nodes are removed, but they are still in the
graph and their data is accessible.

**Arguments**
  * `listener` `function (evt: {nodes: NodeList})`

## `ogma.events.onClick(listener)` **DEPRECATED**

Deprecated, use `click` event instead..

**Arguments**
  * `listener` `function(evt: {x: number, y: number, target: InputTarget, button: MouseButton, source: InputSource, domEvent: Event})`

## `ogma.events.onDoubleClick(listener)` **DEPRECATED**

Deprecated, use `doubleclick` event instead..

**Arguments**
  * `listener` `function(evt: {x: number, y: number, target: InputTarget, button: MouseButton, source: InputSource, domEvent: Event})`

## `ogma.events.onDragEnd(listener)` **DEPRECATED**

Deprecated, use `dragEnd` event instead.
Triggers the specified function when the user releases a mouse button, if a `onDragStart` has been emitted before.

**Arguments**
  * `listener` `function(evt: {x: number, y: number, target: InputTarget, button: MouseButton, source: InputSource, domEvent: Event})` If a node or edge was under the cursor when the first `onDragStart` event was emitted, it is passed as the `target` property.

## `ogma.events.onDragProgress(listener)` **DEPRECATED**

Deprecated, use `dragProgress` event instead.
Triggers the specified function every time the user moves the mouse after a `onDragStart` event has been emitted,
as long as the user doesn't release the mouse.

**Arguments**
  * `listener` `function(evt: {x: number, y: number, target: InputTarget, button: MouseButton, source: InputSource, domEvent: Event})` If a node or edge was under the cursor when the first `onDragStart` event was emitted, it is passed as the `target` property.

## `ogma.events.onDragStart(listener)` **DEPRECATED**

Deprecated, use `dragStart` event instead..

**Arguments**
  * `listener` `function(evt: {x: number, y: number, target: InputTarget, button: MouseButton, source: InputSource, domEvent: Event})`

## `ogma.events.onDrop(listener)` **DEPRECATED**

Deprecated, use `drop` event instead.
Triggered when the user drops an element into the Ogma container. Note that x and y arguments are Graph coordinates.

**Arguments**
  * `listener` `function(evt: {domEvent: Event, x: number, y: number})`

## `ogma.events.onEdgeDataChange(listener)` **DEPRECATED**

Deprecated, use `updateEdgeData` event instead.
Trigger the specified function when the data of some nodes is updated.

**Arguments**
  * `listener` `function(evt: {changes: Array<{property: PropertyPath, edges: EdgeList, previousValues: Array<any>, newValues: Array<any>}>})`

## `ogma.events.onEdgesAdded(listener)` **DEPRECATED**

Deprecated, use `addEdges` event instead.
Triggers the specified function when some edges are added to the graph.

**Arguments**
  * `listener` `function (evt: {edges: EdgeList})`

## `ogma.events.onEdgesRemoved(listener)` **DEPRECATED**

Deprecated, use `removeEdges` event instead.
Triggers the specified function when some edges are removed from the graph.

**Arguments**
  * `listener` `function (evt: {edges: EdgeList})`

## `ogma.events.onEdgesSelected(listener)` **DEPRECATED**

Deprecated, use `edgesSelected` event instead.
Triggers the specified function when some edges are selected.

**Arguments**
  * `listener` `function(evt: {edges: EdgeList})`

## `ogma.events.onEdgesUnselected(listener)` **DEPRECATED**

Deprecated, use `edgesUnselected` event instead.

**Arguments**
  * `listener` `function(evt: {edges: EdgeList})`

## `ogma.events.onGeoModeDisabled(listener)` **DEPRECATED**

Deprecated, use `geoDisabled` event instead.
Triggered when the geo mode is switched off

**Arguments**
  * `listener` `function()`

## `ogma.events.onGeoModeEnabled(listener)` **DEPRECATED**

Deprecated, use `geoEnabled` event instead.
Triggered when the geo mode is activated

**Arguments**
  * `listener` `function()`

## `ogma.events.onGeoModeLoaded(listener)` **DEPRECATED**

Deprecated, use `geoLoaded` event instead.
Triggered when the background map images are loaded

**Arguments**
  * `listener` `function()`

## `ogma.events.onGestureEnd(listener)` **DEPRECATED**

Deprecated, use `gestureEnd` event instead.
Triggers the specified function when the user stop touching the screen with two fingers.

**Arguments**
  * `listener` `function(evt: {domEvent: Event})`

## `ogma.events.onGestureProgress(listener)` **DEPRECATED**

Deprecated, use `gestureProgress` event instead.
Triggers the specified function when the users moves two fingers.

**Arguments**
  * `listener` `function(evt: {x: number, y: number, scale: number, angle: number, dx: number, dy: number, domEvent: Event})`

## `ogma.events.onGestureStart(listener)` **DEPRECATED**

Deprecated, use `gestureStart` event instead.
Triggers the specified function when the user touch the screen with two fingers.

**Arguments**
  * `listener` `function(evt: {domEvent: Event})`

## `ogma.events.onGraphCleared(listener)` **DEPRECATED**

Deprecated, use `clearGraph` event instead.
Triggers the specified function when ogma.clearGraph is called.

**Arguments**
  * `listener` `function (evt: {nodes: NodeList, edges: EdgeList})`

## `ogma.events.onHover(listener)` **DEPRECATED**

Deprecated, use `mouseover` event instead.
Triggers the specified function when a node or edge is hovered.

**Arguments**
  * `listener` `function(evt: {x: number, y: number, target: InputTarget, domEvent: Event})`

## `ogma.events.onLayoutComplete(listener)` **DEPRECATED**

Deprecated, use `layoutEnd` event instead.

**Arguments**
  * `listener` `function(evt: { name: string, ids: Array<NodeId>, positions: { before: Array<{x: number, y: number}>, after: Array<{x: number, y: number}>}})`

## `ogma.events.onLayoutComputed(listener)` **DEPRECATED**

Deprecated, use `layoutComputed` event instead.

This event is fired after the layout algorithm has finished the calculations,
but before the positions are applied. Use it for UI interactions, because if
you would add position manipulations into the listener, they can interfere
with the layout results.

**Arguments**
  * `listener` `function(payload: { name: string })`

## `ogma.events.onLayoutStart(listener)` **DEPRECATED**

Deprecated, use `layoutStart` event instead.

**Arguments**
  * `listener` `function(evt: { name: string, ids: Array<NodeId>})`

## `ogma.events.onMouseButtonDown(listener)` **DEPRECATED**

Deprecated, use `mousedown` event instead..

**Arguments**
  * `listener` `function(evt: {x: number, y: number, target: InputTarget, button: MouseButton, source: InputSource, domEvent: Event})`

## `ogma.events.onMouseButtonUp(listener)` **DEPRECATED**

Deprecated, use `mouseup` event instead..

**Arguments**
  * `listener` `function(evt: {x: number, y: number, target: InputTarget, button: MouseButton, source: InputSource, domEvent: Event})`

## `ogma.events.onMouseMove(listener)` **DEPRECATED**

Deprecated, use `mousemove` event instead..

**Arguments**
  * `listener` `function(evt: {x: number, y: number, dx: number, dy: number, source: InputSource, domEvent: Event})`

## `ogma.events.onMouseWheel(listener)` **DEPRECATED**

Deprecated, use `mousewheel` event instead.
Triggers the specified function when the user uses the mouse wheel.

**Arguments**
  * `listener` `function(evt: {x: number, y: number, delta: number, domEvent: Event})` `delta` is a number between -1 and 1.

## `ogma.events.onNodeDataChange(listener)` **DEPRECATED**

Deprecated, use `updateNodeData` event instead.
Trigger the specified function when the data of some nodes is updated.

**Arguments**
  * `listener` `function(evt: {changes: Array<{property: PropertyPath, nodes: NodeList, previousValues: Array<any>, newValues: Array<any>}>})`

## `ogma.events.onNodeDragEnd(listener)` **DEPRECATED**

Deprecated, use `nodesDragEnd` event instead.
Triggered when the user stop dragging some nodes.

**Arguments**
  * `listener` `function(evt: {nodes: NodeList, start: Array<{x: number, y: number}>, end: Array<{x: number, y: number}>})`

## `ogma.events.onNodeDragProgress(listener)` **DEPRECATED**

Deprecated, use `nodesDragProgress` event instead.
Triggered when the user drags some nodes.

**Arguments**
  * `listener` `function(evt: {nodes: NodeList, dx: number, dy: number })`

## `ogma.events.onNodeDragStart(listener)` **DEPRECATED**

Deprecated, use `nodesDragStart` event instead.
Triggered when the user starts to drag some nodes.

**Arguments**
  * `listener` `function(evt: {nodes: NodeList})`

## `ogma.events.onNodesAdded(listener)` **DEPRECATED**

Deprecated, use `addNodes` event instead.
Triggers the specified function when some nodes are added to the graph.

**Arguments**
  * `listener` `function (evt: {nodes: NodeList})`

## `ogma.events.onNodesConnected(listener)` **DEPRECATED**

Deprecated, use `connectNodes` event instead.
Trigger the specified function when two nodes are connected using the module.

**Arguments**
  * `listener` `function(evt: {source: Node, target: Node, edge: Edge })`

## `ogma.events.onNodesRemoved(listener)` **DEPRECATED**

DEPRECATED: see #Event:-removeNodes DEPRECATED see #Event:-
Triggers the specified function when some nodes are removed from the graph.

**Arguments**
  * `listener` `function (evt: {nodes: NodeList})`

## `ogma.events.onNodesSelected(listener)` **DEPRECATED**

Deprecated, use `nodesSelected` event instead.
Triggers the specified function when some nodes are selected.

**Arguments**
  * `listener` `function(evt: {nodes: NodeList})`

## `ogma.events.onNodesUnselected(listener)` **DEPRECATED**

Deprecated, use `nodesUnselected` event instead.
Triggers the specified function when some nodes are removed from the selection.

**Arguments**
  * `listener` `function(evt: {nodes: NodeList})`

## `ogma.events.onRendererStateChange(listener)` **DEPRECATED**

Deprecated, use `rendererStateChange` event instead.
Triggered when the renderer is requested, successfully initialized or encounters an error.

**Arguments**
  * `listener` `function (evt: {type: RendererType, state: RendererState, code: RendererErrorCode, message: string})`

## `ogma.events.onTooltipHidden(listener)` **DEPRECATED**

Deprecated, use `tooltipHide` event instead.
Triggers the specified function when a tooltip is hidden.

**Arguments**
  * `listener` `function (evt: {tooltip: HTMLElement})`

## `ogma.events.onTooltipShown(listener)` **DEPRECATED**

Deprecated, use `tooltipShow` event instead.
Triggers the specified function when a tooltip is shown.

**Arguments**
  * `listener` `function (evt: {tooltip: HTMLElement})`

## `ogma.events.onTransformationDestroyed(listener)` **DEPRECATED**

Deprecated, use `transformationDestroyed` event instead.
Triggered when a transformation  is destroyed

**Arguments**
  * `listener` `function({ target: Transformation }):void`

## `ogma.events.onTransformationDisabled(listener)` **DEPRECATED**

Deprecated, use `transformationDisabled` event instead.
Triggered when a transformation is disabled

**Arguments**
  * `listener` `function({ target: Transformation }):void`

## `ogma.events.onTransformationEnabled(listener)` **DEPRECATED**

Deprecated, use `transformationEnabled` event instead.
Triggered when a transformation is activated

**Arguments**
  * `listener` `function({ target: Transformation }):void`

## `ogma.events.onTransformationRefreshed(listener)` **DEPRECATED**

Deprecated, use `transformationRefresh` event instead.
Triggered when a transformation index is set

**Arguments**
  * `listener` `function({ target: Transformation }):void`

## `ogma.events.onTransformationSetIndex(listener)` **DEPRECATED**

Deprecated, use `transformationSetIndex` event instead.
Triggered when a transformation index is set

**Arguments**
  * `listener` `function({ target: Transformation, index: number }):void`

## `ogma.events.onUnhover(listener)` **DEPRECATED**

Deprecated, use `mouseOut` event instead.
Triggers the specified function when a node or edge stops being hovered.

**Arguments**
  * `listener` `function(evt: {x: number, y: number, target: InputTarget, domEvent: Event})`

## `ogma.events.onViewChanged(listener)` **DEPRECATED**

Deprecated, use `viewChanged` event instead. is finished.

**Arguments**
  * `listener` `function()`

## `ogma.events.onZoomProgress(listener)` **DEPRECATED**

Deprecated, use `cameraZoom` event instead.
Triggers the specified function when zoom animation is in progress

**Arguments**
  * `listener` `function()`

# Ogma.export

Export API: functions to export graphs to various formats.

## `ogma.export.csv(parameters)`

**Arguments**
  * `parameters` `object|"nodes"|"edges"`
      * `dataProperties`(optional) `Array<PropertyPath>` Data properties to export. If not specified, exports all data properties.
      * `download`(optional) `boolean`[default: true] If true, the user will be prompted a modal window so he can download the exported graph.
      * `edgeData`(optional) `function (data: any): any` Given an edge data in input, must return the object to export as data. By default export the whole edge data untouched.
      * `edges`(optional) `EdgeCollection` Edges to export. By default export all the edges (if `what` is `"edges"`).
      * `filename`(optional) `string`[default: "graph.csv"] If `download` is true, the default name for the downloaded file.
      * `filter`(optional) `Filter`[default: "visible"] Indicates what elements to export.
      * `nodeData`(optional) `function (data: any): any` Given a node data in input, must return the object to export as data. By default export the whole node data untouched.
      * `nodes`(optional) `NodeCollection` Nodes to export. By default export all the nodes (if `what` is `"nodes"`).
      * `separator`(optional) `string`[default: ","] Column separator
      * `textSeparator`(optional) `'"'|"'"`[default: &#039;"&#039;] String used to surround strings. Can only be `"` or `'`.
      * `what` `"nodes"|"edges"` Indicates if nodes or edges should be exported.

**Returns**
  *  `Promise<string>`

## `ogma.export.gexf([parameters])`

**Arguments**
  * `parameters`(optional) `object`
      * `creator`(optional) `string` Name of the creator, that will be specified in the output file
      * `description`(optional) `string` Description of the graph, that will be specified in the output file
      * `download`(optional) `boolean`[default: true] If true, the user will be prompted a modal window so he can download the exported graph.
      * `edgeData`(optional) `function (data: any): any` Given an edge data in input, must return the object to export as data. By default export the whole edge data untouched.
      * `edges`(optional) `EdgeCollection` Edges to export. By default export all the edges.
      * `filename`(optional) `string`[default: "graph.gexf"] If `download` is true, the default name for the downloaded file.
      * `filter`(optional) `Filter`[default: "visible"] Indicates what elements to export.
      * `nodeData`(optional) `function (data: any): any` Given a node data in input, must return the object to export as data. By default export the whole node data untouched.
      * `nodes`(optional) `NodeCollection` Nodes to export. By default export all the nodes.
      * `styles`(optional) `"all"|"none"|"original"`[default: "all"] Indicates what styles (color, shape, size, text) should be exported: `'all'` for what is visually displayed, `'none'` for no style and `'original'` for the values provided at initialization.

**Returns**
  *  `Promise<string>`

## `ogma.export.gif([options])`

**Arguments**
  * `options`(optional) `ImageExportOptions`

**Returns**
  *  `Promise<string>` The argument of the Promise  is the data url of the output image

## `ogma.export.graphml([parameters])`

**Arguments**
  * `parameters`(optional) `object`
      * `directedEdges`(optional) `boolean`[default: true] Indicates in the output file if the edges are directed or not
      * `download`(optional) `boolean`[default: true] If true, the user will be prompted a modal window so he can download the exported graph.
      * `edgeData`(optional) `function (data: any): any` Given an edge data in input, must return the object to export as data. By default export the whole edge data untouched.
      * `edges`(optional) `EdgeCollection` Edges to export. By default export all the edges.
      * `filename`(optional) `string`[default: "graph.graphml"] If `download` is true, the default name for the downloaded file.
      * `filter`(optional) `Filter`[default: "visible"] Indicates what elements to export.
      * `graphId`(optional) `string`[default: "G"] Id of the graph to write in the output file
      * `nodeData`(optional) `function (data: any): any` Given a node data in input, must return the object to export as data. By default export the whole node data untouched.
      * `nodes`(optional) `NodeCollection` Nodes to export. By default export all the nodes.
      * `styles`(optional) `"all"|"none"|"original"`[default: "all"] Indicates what styles (color, shape, size, text) should be exported: `'all'` for what is visually displayed, `'none'` for no style and `'original'` for the values provided at initialization.

**Returns**
  *  `Promise<string>`

## `ogma.export.jpg([options])`

**Arguments**
  * `options`(optional) `ImageExportOptions`

**Returns**
  *  `Promise<string>` The argument of the Promise  is the data url of the output image

## `ogma.export.json([parameters])`

**Arguments**
  * `parameters`(optional) `object`
      * `anonymize`(optional) `boolean`[default: false] If true, the exported graph will be anonimized (i.e. all the nodes
and edges will be exported without data). All the node and edge attributes will be exported as well.
Equivalent to
ogma.export.json({
  nodeAttributes: 'all',
  edgeAttributes: 'all',
  filter: 'all',
  nodeData: () => null,
  edgeData => null
});
.
      * `download`(optional) `boolean`[default: true] If true, the user will be prompted a modal window so he can download the exported graph.
      * `edgeAttributes`(optional) `Array<PropertyPath>|"all"`[=[&#039;color&#039;, &#039;width&#039;, &#039;text&#039;]] List of edge attributes to export. By default, export color, text and width.
      * `edgeData`(optional) `function (data: any): any` Given an edge data in input, must return the object to export as data. By default export the whole edge data untouched.
      * `filename`(optional) `string`[default: "graph.json"] If `download` is true, the default name for the downloaded file.
      * `filter`(optional) `Filter|{nodes: NodeCollection, edges: EdgeCollection}`[default: "visible"] Indicates what elements to export.
      * `nodeAttributes='x',` `Array<[PropertyPath>|"all"` 'y', 'color', 'radius', 'shape', 'text']] List of node attributes to export. By default, export position, color, shape, text and radius.
      * `nodeData`(optional) `function (data: any): any` Given a node data in input, must return the object to export as data. By default export the whole node data untouched.
      * `pretty`(optional) `boolean`[default: false] Indicates if the output should be properly indented.

**Returns**
  *  `Promise<RawGraph>`

## `ogma.export.png([options])`

**Arguments**
  * `options`(optional) `ImageExportOptions`

**Returns**
  *  `Promise<string>` The argument of the Promise  is the data url of the output image

## `ogma.export.svg([parameters])`

**Arguments**
  * `parameters`(optional) `object`
      * `background`(optional) `Color` Color of the background
      * `badges`(optional) `boolean`[default: true] Whether or not to export badges
      * `clip`(optional) `boolean`[default: false] Whether to clip the exported image to the current Ogma viewport.
      * `download`(optional) `boolean`[default: true] If true, the user will be prompted a modal window so he can
                                               download the exported graph.
      * `embedFonts`(optional) `boolean`[default: false] Whether or not to embed custom fonts as base64
                                               (works in viewers and browsers, but in order to edit
                                               you will have to install the fonts on your machine anyway).
                                               Otherwise the custom fonts will be just linked in the file.
      * `filename`(optional) `string`[default: "graph.svg"] If `download` is true, the default name for the downloaded file.
      * `groupSemantically`(optional) `boolean`[default: true] Whether or not group the elements by type (nodes and edges).
If true expect the z-index of the texts to be different than in the visualization.
      * `height`(optional) `number` If not specified, the height of the canvas will be used.
      * `images`(optional) `boolean`[default: true] Indicates if images should be exported.
      * `margin`(optional) `number`[default: 10] Additional margin.
      * `prefix`(optional) `string`[default: &#039;ogma&#039;] Prefix for the entity class names. For example, elements belonging
                                               to a node are grouped into an SVG group with the class `ogma-node`
                                               and an attribute `data-node-id` with the (escaped) node id. The word
                                               `ogma` in these class names can be replaced by a custom string.
      * `texts`(optional) `boolean`[default: true] Whether or not to export texts
      * `width`(optional) `number` If not specified, the width of the canvas will be used.

**Returns**
  *  `Promise<string>` The argument is the SVG string

## `ogma.export.tiff([options])`

**Arguments**
  * `options`(optional) `ImageExportOptions`

**Returns**
  *  `Promise<string>` The argument of the Promise is the data url of the output image

## `ogma.export.xlsx([parameters])`

Requires the xlsx library to be included (if browser) or to be available through 'require' (if Node.js)

**Arguments**
  * `parameters`(optional) `XLSXExportOptions|"nodes"|"edges"`
      * `dataProperties`(optional) `Array<PropertyPath>` Data properties to export. If not specified, exports all data properties. Deprecated : use `nodeData` and `edgeData` instead.
      * `download`(optional) `boolean`[default: true] If true, the user will be prompted a modal window so he can download the exported graph. Browser only.
      * `edgeData`(optional) `function(data: any, allTabs: Array<string>): object|Array<object>|undefined` Indicates how to format the edge data for the tab: each key of the returned `object` is used as `columnName` and its value as `columnValue`. In case of multiple tabs for the edge, an array of objects will be expected.
The `allTabs` array is passed to give some context to the formatting. The default is use the data object "flatten" all his properties. When set it overwrites `dataProperties` configuration.
      * `edges`(optional) `EdgeCollection` Edges to export. By default export all the edges. When set it overwrites "filter" configuration.
      * `filename`(optional) `string`[default: "graph.xlsx"] If `download` is true, the default name for the downloaded file.
      * `filter`(optional) `Filter`[default: "visible"] Indicates what elements to export.
      * `nodeData`(optional) `function(data: any, allTabs: Array<string>): object|Array<object>|undefined` Indicates how to format the node data for the tab: each key of the returned `object` is used as `columnName` and its value as `columnValue`. In case of multiple tabs for the node, an array of objects will be expected.
The `allTabs` array is passed to give some context to the formatting. The default is use the data object "flatten" all his properties. When set it overwrites `dataProperties` configuration.
      * `nodes`(optional) `NodeCollection` Nodes to export. By default export all the nodes. When set it overwrites "filter" configuration.
      * `tab`(optional) `object` Indicates how to name the tabs in the XSLX tabs. When not defined it will exports a "nodes" and a "edges" tab.
  **Note**: tab names cannot contain some characters in Excel, for more information please see: naming conventions for worksheets .
          * `edges `(optional) `function(edge: Edge): string|Array<string>|undefined`[default: (edge: Edge) => "edges"] The returned string indicates the name of the tab (or tabs) to use for the edge. When `undefined` is returned it defaults to "edges".
          * `nodes `(optional) `function(node: Node): string|Array<string>|undefined`[default: (node: Node) => "nodes"] The returned string indicates the name of the tab (or tabs) to use for the node. When `undefined` is returned it defaults to "nodes".

      * `what`(optional) `"nodes"|"edges"` If a value is specified, only nodes or edges will be exported, not both.

**Returns**
  *  `Promise<Blob>`

# Ogma.generate

Graph generators: functions to generate graphs with specific properties to test algorithms or display features.

## `ogma.generate.balancedTree([options])`

Generates a simple balanced tree.
Source: https://github.com/gka/randomgraph.js (license: public domain)

**Arguments**
  * `options`(optional) `object`
      * `children`(optional) `number`[default: 2] The number of children each node has.
      * `height`(optional) `number`[default: 3] The height of the tree.

**Returns**
  *  `Promise<RawGraph>`

## `ogma.generate.barabasiAlbert([options])`

Generates a scale-free graph using preferntial-attachment mechanism.
See Barabási–Albert model (Wikipedia)

**Arguments**
  * `options`(optional) `object`
      * `m`(optional) `number`[default: 1] M  > 0 && m <= m0
      * `m0`(optional) `number`[default: 5] M0 > 0 && m0 <  nodes
      * `nodes`(optional) `number`[default: 40] Number of nodes in the graph.
      * `scale`(optional) `number`[default: 100] Scale > 0 Scale of the space used by graph.

**Returns**
  *  `Promise<RawGraph>`

## `ogma.generate.erdosRenyi([options])`

Generates an Erdős–Rényi graph. Call it with options (n,p) or (n,m).
Source: https://github.com/gka/randomgraph.js (license: public domain)
See Erdős–Rényi model (Wikipedia)

**Arguments**
  * `options`(optional) `object`
      * `edges`(optional) `number` The number of edges. If specified, `p` must not be specified.
      * `nodes`(optional) `number`[default: 20] The number of nodes.
      * `p`(optional) `number`[default: 0.1] The probability [0..1] of a edge between any two nodes. If specified, `edges` must not be specified.

**Returns**
  *  `Promise<RawGraph>`

## `ogma.generate.grid([options])`

Generates a grid.

**Arguments**
  * `options`(optional) `object`
      * `columnDistance`(optional) `number`[default: 20] Distance between two columns of nodes
      * `columns`(optional) `number`[default: 4] The number of columns in the graph.
      * `rowDistance`(optional) `number`[default: 20] Distance between two rows of nodes
      * `rows`(optional) `number`[default: 4] The number of rows in the graph.
      * `xmin`(optional) `number`[default: 0] Start X coordinate for the grid
      * `ymin`(optional) `number`[default: 0] Start Y coordinate for the grid.

**Returns**
  *  `Promise<RawGraph>`

## `ogma.generate.path([options])`

Generates a path.

**Arguments**
  * `options`(optional) `object`
      * `length`(optional) `number`[default: 5] Number of nodes.

**Returns**
  *  `Promise<RawGraph>`

## `ogma.generate.random([options])`

Generates a random graph.

**Arguments**
  * `options`(optional) `object`
      * `edges`(optional) `number`[default: 10] Number of edges.
      * `nodes`(optional) `number`[default: 10] Number of nodes.

**Returns**
  *  `Promise<RawGraph>`

## `ogma.generate.randomTree([options])`

Generates a random tree graph.

**Arguments**
  * `options`(optional) `object`
      * `height`(optional) `number`[default: 100] Height of the space to generate graph in
      * `nodes`(optional) `number`[default: 50] Number of nodes.
      * `width`(optional) `number`[default: 100] Width of the space to generate graph in
      * `x`(optional) `number`[default: 0] X of the center
      * `y`(optional) `number`[default: 0] Y of the center

**Returns**
  *  `Promise<RawGraph>`

# Ogma.geo

Geographical mode API : allows to display nodes which have geographical coordinates (latitude and longitude) on a map.

## `ogma.geo.disable([options])`

Disables geo layout

**Arguments**
  * `options`(optional) `GeoModeOptions`

**Returns**
  *  `Promise<void>`

## `ogma.geo.enable([options])`

Enables geo mode layout

**Arguments**
  * `options`(optional) `GeoModeOptions`

**Returns**
  *  `Promise<void>`

## `ogma.geo.enabled()`

Check whether geographical mode is enabled.

**Returns**
  *  `boolean`

## `ogma.geo.getCenter()`

Returns current map position. Returns undefined when the Geo mode is disabled.

**Returns**
  *  `GeoCoordinate`

## `ogma.geo.getMap()`

Get the underling map object (a Leaflet Map instance). Returns null when the Geo mode is disabled.

**Returns**
  *  `Map`

## `ogma.geo.getOptions()`

Get module settings.

**Returns**
  *  `GeoModeOptions`

## `ogma.geo.getUnprojectedCoordinates([selector])`

Returns underlying X and Y positions for the nodes that are currently
handled by the geo-mode.

**Arguments**
  * `selector`(optional) `Array<NodeId>|Filter|Array<Node>`[default: "visible"]

**Returns**
  *  `Array<{x: number, y: number}>`

## `ogma.geo.getView()`

Returns current map position. Returns undefined when the Geo mode is disabled.

**Returns**
  *  `MapPosition`

## `ogma.geo.getZoom()`

Returns current map zoom level. Returns undefined when the Geo mode is disabled.

**Returns**
  *  `number`

## `ogma.geo.resetCoordinates()`

Reset geographical coordinates of the nodes to the initial values

## `ogma.geo.runLayout(parameters)`

Helper method to get the geo coordinates for the nodes that don't have them,
if you want to position them on the map in a specific way using one of the layouts.

**Arguments**
  * `parameters` `object`
      * `center`(optional) `GeoCoordinate` A coordinate to use as the center of the layout. If not provided, the algorithm will try to use the average of the coordinates of the nodes that have geo coordinates. If none of them do and the center is not provided explicitly, an error will be thrown.
      * `latitudePath`(optional) `PropertyPath`[default: &#039;latitude&#039;] Node path which contains the latitude.
      * `longitudePath`(optional) `PropertyPath`[default: &#039;longitude&#039;] Node path which contains the longitude.
      * `nodes`(optional) `NodeList` Nodes to layout. If not provided, all nodes will be used.
      * `options` `LayoutOptions` Layout options. Respective layout options can be found in the layout documentation.
      * `type` `'force'|'forceLink'|'hierarchical'|'sequential'|'grid'|'radial'|'concentric'` Layout name

**Returns**
  *  `Promise<GeoCoordinate[]>`

## `ogma.geo.setCenter(latitude, longitude)`

Centers the map at given coodinates

**Arguments**
  * `latitude` `number`
  * `longitude` `number`

## `ogma.geo.setOptions([options])`

Update module settings

**Arguments**
  * `options`(optional) `GeoModeOptions`

## `ogma.geo.setView(latitude, longitude, zoom)`

Set map view - coordinates and zoom level

**Arguments**
  * `latitude` `number`
  * `longitude` `number`
  * `zoom` `number`

## `ogma.geo.setZoom(zoom)`

Sets zoom level of the map

**Arguments**
  * `zoom` `number`

## `ogma.geo.toggle([options])`

Toggles geo mode. Useful when you don't want to store information about
whether the mode was on or off(e.g. with an UI switcher).

**Arguments**
  * `options`(optional) `GeoModeOptions`

**Returns**
  *  `Promise<void>`

# Ogma.keyboard

Keyboard API: functions to listen to keyboard events.

## `ogma.keyboard.isKeyPressed(key)`

Indicates if the specified key is pressed.

**Arguments**
  * `key` `KeyName|KeyCode` Key name or key code indicating the key to
check.

**Returns**
  *  `boolean`

## `ogma.keyboard.resetKeys()`

Resets the stored values for the keys that the user has pressed. Useful to
ensure that after a certain shortcut the next combination will be detected
properly, even if the user made a mistake. Also use it in the browsers
which do not report loss of focus when a dialog window is open.

# Ogma.layers

Layers API: functions to add custom annotation layers to your graph visualization.

## `ogma.layers.addCanvasLayer(draw[, index][, options])`

Add a canvas layer. Useful to perform drawings in sync with the view. In the
drawing function you are given the CanvasRenderingContext2D, that is
automatically scaled and translated to be in sync with the graph. So you
can simply use graph coordinates to draw shapes and text in it.
See our "Layers" examples for the code snippets.

**Arguments**
  * `draw` `DrawingFunction` The function drawing on the canvas in
the graph space.
  * `index`(optional) `number` The optional index of the layer.
  * `options`(optional) `CanvasLayerOptions` The optional layer options

**Returns**
  *  `CanvasLayer` Returns the canvas layer object.

## `ogma.layers.addLayer(element[, index])`

Add an HTML element as an overlay to Ogma at the specifided index (if provided).
This is the most basic way of adding your custom HTML elements to the
Ogma canvas, they will be positioned at a coordinate (in graph space) and
kept in sync with the camera movement, but not scaled.

**Arguments**
  * `element` `HTMLElement | string` The HTML element attached to the layer. Can be an HTML string.
  * `index`(optional) `number` The optional index of the layer.

**Returns**
  *  `Layer` Returns the layer object.

## `ogma.layers.addOverlay(options[, index])`

Adds an overlay layer to Ogma, that will update its position and scale in
sync with the graph. Ideal for images or complex annotations of the nodes.

**Arguments**
  * `options` `OverlayOptions` HTML element
provided with its affine transformation in the graph space.
  * `index`(optional) `number` The optional index of the layer.

**Returns**
  *  `Overlay` Returns the layer object.

## `ogma.layers.addSVGLayer([options][, index])`

Add a SVG layer. Useful to perform drawings in sync with the view. In the
drawing function, the svg is automatically scaled and translated to be in sync with the graph.
So you can simply use graph coordinates to draw shapes and text in it.
See our "Layers" examples for the code snippets.

**Arguments**
  * `options`(optional) `SVGLayerOptions` The optional layer options
  * `index`(optional) `number` The optional index of the layer.

**Returns**
  *  `SVGLayer` Returns the canvas layer object.

# Ogma.layouts

Layout API: algorithms to calculate graph layouts.

## `ogma.layouts.concentric(params)`

Concentric layout. This layout takes a base node as parameter and organizes the graph so the nodes close to the
selected node are close to it spatially.

**Arguments**
  * `params` `object` See LayoutOptions for common layout options.
      * `allowOverlap`(optional) `boolean`[default: false] Specifies if nodes are allowed to overlap.
      * `centerX`(optional) `number` X coordinate where the central node must be moved, if different from the central node X
      * `centerY`(optional) `number` Y coordinate where the central node must be moved, if different from the central node Y
      * `centralNode` `NodeId|Node` Id of the central node
      * `circleHopRatio`(optional) `number`[default: 5] If `allowOverlap` is false, specified the space between each ring, relative to the highest node size.
      * `clockwise`(optional) `boolean`[default: true] Specifies if the nodes must be ordered clockwise.
      * `duration`(optional) `number`[default: 300] Duration of the animation when the graph is updated
      * `easing`(optional) `Easing`[default: &#039;cubicOut&#039;] Easing function used during the animation
      * `edges`(optional) `EdgeId[]|EdgeList` List of affected edges. If nothing provided, the adjacent edges to the node list is used.
      * `locate`(optional) `boolean|LocateOptions`[default: false] Center on the graph bounding box when the layout is complete. You can also provide padding.
      * `nodes`(optional) `NodeId[]|NodeList` List of affected nodes. If nothing provided, the whole graph will be used. If edges are provided too, then this list will be augmented with reached nodes from the passed edge list.
      * `onEnd`(optional) `function(): void` Function called after the last graph update
      * `onSync`(optional) `function(): void` Function called every time the graph is updated
      * `skipTextDrawing`(optional) `boolean`[default: true] Skip drawing labels during the layout. Improves performance and user experience.
      * `sortBy`(optional) `string` Indicates the property from which the nodes must be sorted, or
                                                `'random'`. You can use `'radius'`, `'degree'` or custom data
                                                attributes.
      * `useWebWorker`(optional) `boolean`[default: true] Indicates if the layout should be computed inside a web worker.

**Returns**
  *  `Promise<void>`

## `ogma.layouts.force(params)`

Force-directed layout.

**Arguments**
  * `params` `object` See LayoutOptions for common layout options.
      * `alignSiblings`(optional) `boolean`[default: false] Align nodes that are linked to the same two nodes only. It enhances
                                              readability. This operation is performed once the main layout is
                                              finished.
      * `autoStop`(optional) `autoStop`[default: false] Stop layout earlier if the algorithm decides that it has converged to
                                          a stable configuration. It can make the algorithm run much faster and
                                          would require fewer iterations if you are restarting the layout after
                                          it has already converged.
      * `charge`(optional) `number`[default: 10] Distance factor between nodes. A greater value increases the distance.
      * `cx`(optional) `number` X coordinate of the layout mass center.
      * `cy`(optional) `number` Y coordinate of the layout mass center.
      * `duration`(optional) `number`[default: 300] Duration of the animation when the graph is updated
      * `easing`(optional) `Easing`[default: &#039;cubicOut&#039;] Easing function used during the animation
      * `edgeLength`(optional) `number`[default: 30] Desired length of an edge connecting 2 nodes.
      * `edgeStrength`(optional) `number`[default: 0.75] Attraction strength. Higher values make edges' attractive force stronger.
      * `edgeWeight`(optional) `function(Edge):number` Use this getter to modify individual weight. 0 means that the edge will be ignored.
      * `edges`(optional) `EdgeId[]|EdgeList` List of affected edges. If nothing provided, the adjacent edges to the node list is used.
      * `elasticity`(optional) `number`[default: 0.9] Node collision elasticity. Smaller values may result in incomplete node
                                            overlap removal. Passing `0` will skip that algorithm pass altogether.
      * `gpu`(optional) `boolean`[default: false] Enable GPU acceleration using WebGL. It makes the algoritm run faster and
                                    produce higher quality layouts. You cannot use Web Workers together with `gpu`
                                    accelearation, so `useWebWorker` is ignored if `gpu: true`.
      * `gravity`(optional) `number`[default: 0.015] Force that attracts nodes to the center of the graph. Center is either
                                            the mass center of the graph or the value defined by `cx` and `cy`.
                                            Greater value makes the layout more compact.
      * `incremental`(optional) `boolean | object`[default: false] Enable the incremental layout using Force layout. When true is uses the default options.
          * `margin `(optional) `number`[default: 5] Will apply Force layout to the group and place the resulting configuration in the closest available position, maintaining a `margin`.
          * `referenceNodes`(optional) `NodeId[]|NodeList`

      * `locate`(optional) `boolean|LocateOptions`[default: false] Center on the graph bounding box when the layout is
                                                     complete. You can also provide padding.
      * `nodeMass`(optional) `function(Node):number` Use this getter to assign individual node masses. Avoid very small masses, as it can lead to numerical instability.
      * `nodes`(optional) `NodeId[]|NodeList` List of affected nodes. If nothing provided, the whole graph will be used. If edges are provided too, then this list will be augmented with reached nodes from the passed edge list.
      * `onEnd`(optional) `function(): void` Function called after the last graph update
      * `onSync`(optional) `function(): void` Function called every time the graph is updated
      * `radiusRatio`(optional) `number`[default: 1.25] Radius ratio is used to allow for small gaps between the nodes while
                                            avoiding the overlapping.
      * `siblingsOffset`(optional) `number`[default: 0.0] Additional offset between the node siblings, so that the distance
                                            to the next node in the row would be r * (1 + siblingsOffset), where r
                                            is the previous node's radius.
      * `skipTextDrawing`(optional) `boolean`[default: true] Skip drawing labels during the layout.
      * `steps`(optional) `number`[default: 300] Iteration steps limit and cooling ratio.
      * `theta`(optional) `number`[default: 0.62] Theta parameter of the Barnes-Hut optimization. Plays the role of
                                            the precision in repulsive forces approximation.
      * `useWebWorker`(optional) `boolean`[default: true] Indicates if the layout should be computed inside a web worker

**Returns**
  *  `Promise<void>`

## `ogma.layouts.forceLink(params[, randomize][, randomizeFactor])`

Force link layout.

**Arguments**
  * `params` `object` See LayoutOptions for common layout options.
      * `alignNodeSiblings`(optional) `boolean`[default: true] Align nodes that are linked to the same two nodes only. It enhances readability. This operation is performed once the main layout is finished.
      * `autoStop`(optional) `boolean`[default: true] The layout stops automatically if true.
      * `avgDistanceThreshold`(optional) `number`[default: 0.01] This is the normal stopping condition of autoStop: true. When the average displacements of nodes is below this threshold, the layout stops.
      * `barnesHutOptimize`(optional) `boolean`[default: false] Should we use the algorithm's Barnes-Hut to improve repulsion's scalability (`O(n²)` to `O(nlog(n))`)? This is useful for large graphs (5000+ nodes) but harmful to small ones.
      * `barnesHutTheta`(optional) `number`[default: 0.5] Theta parameter of the Barnes-Hut optimization.
      * `duration`(optional) `number`[default: 300] Duration of the animation when the graph is updated
      * `easing`(optional) `Easing`[default: &#039;cubicOut&#039;] Easing function used during the animation
      * `edgeWeight`(optional) `function(edge:Edge):number` Use this getter to modify edge weight. 0 means that the edge will be ignored
      * `edgeWeightInfluence`(optional) `number`[default: 0] Increase attraction force between nodes connected with edges of positive weights. Disabled by default.
      * `edges`(optional) `EdgeId[]|EdgeList` List of affected edges. If nothing provided, the adjacent edges to the node list is used.
      * `gravity`(optional) `number`[default: 1] Force which attracts nodes to the center of the graph. A greater value makes the graph more compact.
      * `incremental`(optional) `boolean | object`[default: false] Enable the incremental layout using Force layout. When true is uses the default options.
          * `margin `(optional) `number`[default: 5] Will apply Force layout to the group and place the resulting configuration in the closest available position, maintaining a `margin`.
          * `referenceNodes`(optional) `NodeId[]|NodeList`

      * `iterationsPerRender`(optional) `number`[default: 10] Number of iterations to be run before each update of the graph visualization.
      * `linLogMode`(optional) `boolean`[default: false] Alternative energy model with linear repulsion force and logarithmic attraction force.
      * `locate`(optional) `boolean|LocateOptions`[default: false] Center on the graph bounding box when the layout is complete. You can also provide padding.
      * `maxIterations`(optional) `number`[default: 1000] Set a limit to the number of iterations if autoStop: true.
      * `nodeMass`(optional) `function(node:Node):number` Use this getter to assign node masses. Node degree is passed in for convenience.
      * `nodeSiblingsAngleMin`(optional) `number`[default: 0] Force a minimal angle between aligned nodes (from 0 to PI / 2). Node labels may indeed overlap on horizontally aligned nodes.
      * `nodeSiblingsScale`(optional) `number`[default: 5] Distance multiplier between the aligned nodes.
      * `nodes`(optional) `NodeId[]|NodeList` List of affected nodes. If nothing provided, the whole graph will be used. If edges are provided too, then this list will be augmented with reached nodes from the passed edge list.
      * `onEnd`(optional) `function(): void` Function called after the last graph update
      * `onSync`(optional) `function(): void` Function called every time the graph is updated
      * `outboundAttractionDistribution`(optional) `boolean`[default: false] Attract super-nodes (with many edges) to the outside.
      * `scalingRatio`(optional) `number`[default: 100] Distance factor between nodes. A greater value increases the distance.
      * `skipTextDrawing`(optional) `boolean`[default: true] Skip drawing labels during the layout. Improves performance and user experience.
      * `slowDown`(optional) `number`[default: 1] Reduces the speed of node displacements as the number of iterations increases.
      * `startingIterations`(optional) `number`[default: 10] Number of iterations to be run before the first update of the graph visualization.
      * `strongGravityMode`(optional) `boolean`[default: true] Enable a gravity formula to have a strong effect.
      * `useWebWorker`(optional) `boolean`[default: true] Indicates if the layout should be computed inside a web worker.

  * `randomize`(optional) `string` Whether to randomize the node positions before running the layout. Possible values are `locally` and `globally`. `Locally` means that the node coordinate will be shuffled around its current position, whereas with `globally` it will be assigned a new random value.
  * `randomizeFactor`(optional) `number`[default: 1] [1] Randomization scaling factor.

**Returns**
  *  `Promise<void>`

## `ogma.layouts.grid(params)`

Arrange the nodes in a grid.

**Arguments**
  * `params` `object` See LayoutOptions for common layout options.
      * `colDistance`(optional) `number` Desired distance between the columns of nodes. It should be a positive number and it will be uniformly applied to the whole layout.
      * `cols`(optional) `number` Indicates the desired number of cols. If neither `rows` or `cols` are specified, the layout will attempt to make a square.
      * `duration`(optional) `number`[default: 300] Duration of the animation when the graph is updated
      * `easing`(optional) `Easing`[default: &#039;cubicOut&#039;] Easing function used during the animation
      * `locate`(optional) `boolean|LocateOptions`[default: false] Center on the graph bounding box when the layout is complete. You can also provide padding.
      * `nodes`(optional) `NodeId[]|NodeList` List of affected nodes. If nothing provided, the whole graph will be used
      * `onEnd`(optional) `function(): void` Function called after the last graph update
      * `onSync`(optional) `function(): void` Function called every time the graph is updated
      * `reverse`(optional) `boolean` [false] If `true`, the nodes will be sorted in reverse order.
      * `rowDistance`(optional) `number` Desired distance between the rows of nodes. It should be a positive number and it will be uniformly applied to the whole layout.
      * `rows`(optional) `number` Indicates the desired number of rows. If neither `rows` or `cols` are specified, the layout will attempt to make a square.
      * `skipTextDrawing`(optional) `boolean`[default: true] Skip drawing labels during the layout. Improves performance and user experience.
      * `sortBy`(optional) `PropertyPath` Indicates the property from which the nodes must be sorted. Can also be `'random'`, in which case the node order is randomized. `'degree'` is a possible value, too.
      * `sortFallbackValue`(optional) `any` Use this value for the nodes, for which the sorting attribute is undefined.
      * `useWebWorker`(optional) `boolean`[default: true] Indicates if the layout should be computed inside a web worker.

**Returns**
  *  `Promise<void>`

## `ogma.layouts.hierarchical(params)`

The hierarchical layout positions nodes starting from a root nodes downwards
generating a visual hierarchy based on connectivity.

When the user provides the root nodes then the algorithm positions the cascading nodes
based on their graph-theoretical distance. When root nodes are not provided then the algorithm
works out the best top node in order to reduce the number of layers (depth) of the hierarchy.
It is possibile to impose constraints to the layout in order to set specific layers (depth) for each
node using a numeric `layer` data attribute.
If you want to force a node to be at the top or the bottom of the layering, look at the `roots` and `sinks` parameters.

If there are subgraphs or nodes not reachable from the central node,
they will be layouted separately. It is possibile to control how these subgraphs are positioned.

**Arguments**
  * `params` `object` See LayoutOptions for common layout options.
      * `arrangeComponents`(optional) `"fit"|"grid"|"singleLine"`[default: &#039;fit&#039;] Desired fit for multiple disconnected components: "fit" attempt to optimize the screen space; "grid" adds a special behaviour for isolated nodes, arranging them together in a grid, then fit on the screen; "singleLine" arrange all disconnected components alongside.
      * `componentDistance`(optional) `number`[default: 25] Desired distance between disconnected components
      * `direction`(optional) `"TB"|"BT"|"LR"|"RL"`[default: &#039;TB&#039;] Layout direction: Top-bottom/bottom-top/left-right/right-left.
      * `duration`(optional) `number`[default: 300] Duration of the animation when the graph is updated
      * `easing`(optional) `Easing`[default: &#039;cubicOut&#039;] Easing function used during the animation
      * `edges`(optional) `EdgeId[]|EdgeList` List of affected edges. If nothing provided, the adjacent edges to the node list is used.
      * `gapWidth`(optional) `number`[default: 1] Desidered width of gap spaces between nodes not sequentially next to each other (siblings). This is an advanced parameter, often not required to change, it is similar to the previous "edgeDistance" parameter in Dagre. Expressed in percentage (%) from 1 to 100.
      * `gridDistance`(optional) `number`[default: 50] Desidered distance between isolated nodes when arranged in grid. Used only when "grid" arrangeComponent is enabled.
      * `layer`(optional) `string` Data field defining the layer of the node in the hierarchy
      * `levelDistance`(optional) `number`[default: 50] Desired distance between the layers of layout
      * `locate`(optional) `boolean|LocateOptions`[default: false] Center on the graph bounding box when the layout is complete. You can also provide padding.
      * `nodeDistance`(optional) `number`[default: 5] Desired distance between the nodes on one layer
      * `nodes`(optional) `NodeId[]|NodeList` List of affected nodes. If nothing provided, the whole graph will be used. If edges are provided too, then this list will be augmented with reached nodes from the passed edge list.
      * `onEnd`(optional) `function(): void` Function called after the last graph update
      * `onSync`(optional) `function(): void` Function called every time the graph is updated
      * `roots`(optional) `NodeId[]|NodeList`[=[]] List of nodes to put at the top of the hierarchy
      * `sinks`(optional) `NodeId[]|NodeList`[=[]] List of nodes to put at the bottom of the hierarchy
      * `skipTextDrawing`(optional) `boolean`[default: true] Skip drawing labels during the layout. Improves performance and user experience.

**Returns**
  *  `Promise<void>`

## `ogma.layouts.radial(params)`

Radial (concentric) layout positions nodes around the selected one based on
their graph-theoretical distance (shortest path in the graph, connecting them).
If there are subgraphs or nodes not reachable from the central node,
they will be pushed outwards, but still placed around the layout
in a readable way.

**Arguments**
  * `params` `object` See LayoutOptions for common layout options.
      * `allowOverlap`(optional) `boolean`[default: false] Specifies if nodes are allowed to overlap.
      * `centerX`(optional) `number` X coordinate where the central node must be moved,
                                             if different from the central node X
      * `centerY`(optional) `number` Y coordinate where the central node must be moved,
                                             if different from the central node Y
      * `centralNode` `Node|NodeId` Id of the central node
      * `duration`(optional) `number`[default: 300] Duration of the animation when the graph is updated
      * `easing`(optional) `Easing`[default: &#039;cubicOut&#039;] Easing function used during the animation
      * `epsilon`(optional) `number`[default: 0.001] Layout precision. Smaller number means
                                             better precision but longer computation time
      * `iterationsPerRender`(optional) `number`[default: 20] Layout iterations per update.
      * `locate`(optional) `boolean|LocateOptions`[default: false] Center on the graph bounding box when the layout is complete. You can also provide padding.
      * `maxIterations`(optional) `number`[default: 100] Maximum number of layout sweeps
      * `nodeGap`(optional) `number`[default: 10] Additional gap between the nodes
                                             that belong to one layer
      * `nodes`(optional) `Array<NodeId>|NodeList` List of affected nodes. If nothing provided, the whole graph will be used.
      * `onEnd`(optional) `function(): void` Function called after the last graph update
      * `onSync`(optional) `function(): void` Function called every time the graph is updated
      * `radiusDelta`(optional) `number`[default: 0] You can specify a constant distance
                                             between the layout layers, in this case
                                             `radiusRatio` will be ignored
      * `radiusRatio`(optional) `number`[default: Math.SQRT2] Ratio between the radii
                                             of adjacent concentric layers:
                                             R[n+1] = R[n] × ratio
      * `renderSteps`(optional) `boolean`[default: false] Render intermediate results, before the
                                             algorithm converges. That means sending the
                                             calculated positions every `iterationsPerRender`
                                             iterations.
      * `repulsion`(optional) `number`[default: 1] Increase or decrease the repulsion force between the nodes on
                                             the same levels. Values smaller than 1 will result in more compact
                                              placement along the layers.
      * `skipTextDrawing`(optional) `boolean`[default: true] Skip drawing labels during the layout. Improves performance and user experience.
      * `useWebWorker`(optional) `boolean`[default: true] Indicates if the layout should be computed inside a web worker.

**Returns**
  *  `Promise<void>`

## `ogma.layouts.sequential(params)`

The sequential layout positions nodes starting from a root nodes downwards
generating a visual hierarchy based on connectivity with a costant width.

When the user provides the root nodes then the algorithm positions the cascading nodes
based on their graph-theoretical distance. When root nodes are not provided then the algorithm
works out the best top node in order to reduce the number of layers (depth) of the hierarchy.
It is possibile to impose constraints to the layout in order to set specific layers (depth) for each
node using a numeric `layer` data attribute.
If you want to force a node to be at the top or the bottom of the layering, look at the `roots` and `sinks` parameters.

If there are subgraphs or nodes not reachable from the central node,
they will be layouted separately. It is possibile to control how these subgraphs are positioned.

**Arguments**
  * `params` `object` See LayoutOptions for common layout options.
      * `arrangeComponents`(optional) `"fit"|"grid"|"singleLine"`[default: &#039;fit&#039;] Desired fit for multiple disconnected components: "fit" attempt to optimize the screen space; "grid" adds a special behaviour for isolated nodes, arranging them together in a grid, then fit on the screen; "singleLine" arrange all disconnected components alongside.
      * `componentDistance`(optional) `number`[default: 50] Desired distance between the components in the layout
      * `direction`(optional) `"TB"|"BT"|"LR"|"RL"`[default: &#039;TB&#039;] Layout direction: Top-bottom/bottom-top/left-right/right-left.
      * `duration`(optional) `number`[default: 300] Duration of the animation when the graph is updated
      * `easing`(optional) `Easing`[default: &#039;cubicOut&#039;] Easing function used during the animation
      * `edges`(optional) `EdgeId[]|EdgeList` List of affected edges. If nothing provided, the adjacent edges to the node list is used.
      * `gridDistance`(optional) `number`[default: 50] Desidered distance between isolated nodes when arranged in grid. Used only when "grid" arrangeComponent is enabled.
      * `levelDistance`(optional) `number`[default: 50] Desired distance between the layers of layout
      * `locate`(optional) `boolean|LocateOptions`[default: false] Center on the graph bounding box when the layout is complete. You can also provide padding.
      * `nodeDistance`(optional) `number`[default: 50] Desired distance between the nodes on one layer
      * `nodes`(optional) `NodeId[]|NodeList` List of affected nodes. If nothing provided, the whole graph will be used. If edges are provided too, then this list will be augmented with reached nodes from the passed edge list.
      * `onEnd`(optional) `function(): void` Function called after the last graph update
      * `onSync`(optional) `function(): void` Function called every time the graph is updated
      * `roots`(optional) `NodeId[]|NodeList`[=[]] List of nodes to put at the top of the hierarchy
      * `sinks`(optional) `NodeId[]|NodeList`[=[]] List of nodes to put at the bottom of the hierarchy
      * `skipTextDrawing`(optional) `boolean`[default: true] Skip drawing labels during the layout. Improves performance and user experience.

**Returns**
  *  `Promise<void>`

## `ogma.layouts.stop()`

Stops currently running layout

# Ogma.parse

Import API: functions to import graphs from various formats.

## `ogma.parse.gexf(content)` **DEPRECATED**

Parse a GEXF string and return the raw graph. Use `Ogma.parse.gexfFromUrl` instead.

**Arguments**
  * `content` `string`

**Returns**
  *  `Promise<RawGraph>`

## `ogma.parse.gexfFromUrl(url)` **DEPRECATED**

Fetch and parse a GEXF file and return the raw graph. Use `Ogma.parse.gexfFromUrl` instead.

**Arguments**
  * `url` `string`

**Returns**
  *  `Promise<RawGraph>`

## `ogma.parse.graphml(content[, xmlparser])` **DEPRECATED**

**Arguments**
  * `content` `string` GraphML string
  * `xmlparser`(optional) `DOMParser` DOMParser instance to use for parsing the XML

**Returns**
  *  `Promise<RawGraph>`

## `ogma.parse.graphmlFromUrl(content[, xmlparser])` **DEPRECATED**

**Arguments**
  * `content` `string` GraphML string
  * `xmlparser`(optional) `DOMParser` DOMParser instance to use for parsing the XML

**Returns**
  *  `Promise<RawGraph>`

## `ogma.parse.janus(content)` **DEPRECATED**

Parse the result of a JanusGraph query into an Ogma graph. Use `Ogma.parse.janus` instead.

**Arguments**
  * `content` `object` Response of the gremlin-client library ("gremlin-client")

**Returns**
  *  `Promise<RawGraph>`

## `ogma.parse.json(content[, transform])` **DEPRECATED**

Parse a JSON string and return the raw graph. Use `Ogma.parse.json` instead.

**Arguments**
  * `content` `string`
  * `transform`(optional) `function(json: object | unknown[]): RawGraph` Function
to transform custom JSON format into Ogma's `RawGraph`

**Returns**
  *  `Promise<RawGraph>`

## `ogma.parse.jsonFromUrl(url[, transform])` **DEPRECATED**

Fetch and parse a JSON file and return the raw graph. Use `Ogma.parse.jsonFromUrl` instead.

**Arguments**
  * `url` `string`
  * `transform`(optional) `function(json: object | unknown[]): RawGraph` Function
to transform custom JSON format into Ogma's `RawGraph`

**Returns**
  *  `Promise<RawGraph>`

## `ogma.parse.neo4j(content)` **DEPRECATED**

Parse the result of a Neo4J query into an Ogma graph. Use `Ogma.parse.neo4j` instead.

The parsed user's data will be stored into each Ogma item "data" field with the following structure:

● `neo4jProperties` data field for each item in Ogma,

● `neo4jLabels` field for Neo4j node labels information,

● `neo4jType` field for Neo4j edge types;

**Arguments**
  * `content` `object` Response of the Neo4j Bolt driver ("neo4j-javascript-driver")

**Returns**
  *  `Promise<RawGraph>`

# Ogma.rules

Rules API: functions to create rules for styling nodes and edges.

## `ogma.rules.map(options)`

Create a function that, given a node or edge, returns a value based on a mapping "any data -> any value".

**Arguments**
  * `options` `object`
      * `fallback`(optional) `any|Array<any>` Value(s) to assign when the data value is not specified in the mapping.
      * `field` `PropertyPath` Data property path on which to retrieve the value to be mapped
      * `values`(optional) `{[key: string]: any}` Mapping that associate data value to output value.

**Returns**
  *  `function(elt: Node|Edge): any`

## `ogma.rules.slices(options)`

Create a function that, given a node or edge, returns a value based on a mapping "numerical data -> any value"

**Arguments**
  * `options` `object`
      * `fallback`(optional) `any` Value to assign when the property is not a number.
      * `field` `PropertyPath` Data property path to "slice"
      * `reverse`(optional) `boolean`[default: false] By default low values for data properties are given low output values. Setting this to `true` reverse this behavior.
      * `stops`(optional) `{min: number, max: number}|Array<number>` Indicates the boundaries of the slices. If an object is specified, it must indicates the minimum and maximum values the
data property can have. If it's an array, two consecutive elements indicate the boundaries for a slice. By default, the
slices will be determined using the current minimum and maximum value of the data property.
      * `values` `{nbSlices: number, min: number, max: number}|Array<any>` Indicates the possible output values for the slices. If an object is specified, there will
be `nbSlices` possible output values, with values from `min` to `max` (at regular intervals).
If an array is specified, it indicates the different possible values.

**Returns**
  *  `function(elt: Node|Edge): any`

## `ogma.rules.template(template)`

Returns a function that, given a node or edge, returns a string by replacing a pattern in the string by the
according data property.

**Arguments**
  * `template` `string` String in which "{{foo}}}" will be replaced by the data property "foo".

**Returns**
  *  `function(elt: Node|Edge): any`

# Ogma.schema

Schema API: functions to watch and analyze the data schema of the graph.

## `ogma.schema.watchEdgeNonObjectProperty([options])`

Provide information on the specified edge data property, and notifies when the property is modified.

**Arguments**
  * `options`(optional) `WatcherOptions` Options to use by the watcher

**Returns**
  *  `NonObjectPropertyWatcher`

## `ogma.schema.watchEdgeObjectProperty([options])`

Watch for addition, removal, and updates of the sub-properties of the
specified edge data property.

**Arguments**
  * `options`(optional) `WatcherOptions` Options to use by the watcher

**Returns**
  *  `ObjectPropertyWatcher`

## `ogma.schema.watchNodeNonObjectProperty([options])`

Provide information on the specified node data property, and notifies when
the property is modified.

**Arguments**
  * `options`(optional) `WatcherOptions` Options to use by the watcher

**Returns**
  *  `NonObjectPropertyWatcher`

## `ogma.schema.watchNodeObjectProperty([options])`

Watch for addition, removal, and updates of the sub-properties of the
specified node data property.

**Arguments**
  * `options`(optional) `WatcherOptions` Options to use by the watcher

**Returns**
  *  `ObjectPropertyWatcher`

# Ogma.styles

Ogma functions related to styling the graph and adjusting its visual appearance.

## `ogma.styles.addEdgeRule([selector], rule)`

Add a rule that impacts only edges.

**Arguments**
  * `selector`(optional) `EdgeSelector`
  * `rule` `EdgeAttributesValue`

**Returns**
  *  `StyleRule`

## `ogma.styles.addNodeRule([selector], rule)`

Add a rule that impacts only nodes.

**Arguments**
  * `selector`(optional) `NodeSelector`
  * `rule` `NodeAttributesValue`

**Returns**
  *  `StyleRule`

## `ogma.styles.addRule([options])`

Add a style rule, applying the specified attributes to all nodes & edges that match the specified selector.
The style of a node is re-computed when its degree or data changes, and automatically assigned when a node is added.
Rules are applied one after another. The latest added rule is applied last. Rules are applied before
attributes assigned through `setAttributes`, which are applied before classes.

**Arguments**
  * `options`(optional) `object`
      * `edgeAttributes`(optional) `EdgeAttributesValue|(edge:Edge)=>EdgeAttributesValue` Attributes that will be assigned to the edges.
      * `edgeDependencies`(optional) `EdgeDependencies` (Advanced - see tutorial)
Attributes on which the functions (if any) in the `edgeAttributes` field depend
      * `edgeOutput`(optional) `EdgeOutput` (Advanced - see tutorial)
Edge attributes assigned by the rule. If unspecified, they are inferred from the `edgeAttributes` field.
This field is used together with the dependency fields of other rules/classes to know which rules/classes should be
updated when this rule is updated.
      * `edgeSelector`(optional) `EdgeSelector` Indicates if the rule should be applied to a given edge. If unspecified, the rule is applied to all edges.
      * `nodeAttributes`(optional) `NodeAttributesValue|(node:Node)=>NodeAttributesValue` Attributes that will be assigned to the nodes.
      * `nodeDependencies`(optional) `NodeDependencies` (Advanced - see tutorial)
Attributes on which the functions (if any) in the `nodeAttributes` field depend
      * `nodeOutput`(optional) `NodeOutput` (Advanced - see tutorial)
Node attributes assigned by the rule. If unspecified, they are inferred from the `nodeAttributes` field.
This field is used together with the dependency fields of other rules/classes to know which rules/classes should be
updated when this rule is updated.
      * `nodeSelector`(optional) `NodeSelector` Indicates if the rule should be applied to a given node. If unspecified, the rule is applied to all nodes.

**Returns**
  *  `StyleRule`

## `ogma.styles.createClass(options)`

Create a new class for nodes & edges. Classes are similar to style rules, except they are assigned on an
individual basis instead of according to a selector (assigned only to the nodes/edges that have been
assigned the class with `node.addClass('className')`).

**Arguments**
  * `options` `object`
      * `edgeAttributes`(optional) `EdgeAttributesValue` Attributes applied to edges when they have this class.
      * `edgeDependencies`(optional) `EdgeDependencies` (Advanced - see tutorial)
Attributes on which the functions (if any) in the `edgeAttributes` field depend
      * `edgeOutput`(optional) `EdgeOutput` (Advanced - see tutorial)
Edge attributes assigned by the class. If unspecified, they are inferred from the `edgeAttributes` field.
This field is used together with the dependency fields of other rules/classes to know which rules/classes should be
updated when this class is assigned/removed to/from a edge.
      * `name` `string` Name of the class to be created.
      * `nodeAttributes`(optional) `NodeAttributesValue` Attributes applied to nodes when they have this class.
      * `nodeDependencies`(optional) `NodeDependencies` (Advanced - see tutorial)
Attributes on which the functions (if any) in the `nodeAttributes` field depend
      * `nodeOutput`(optional) `NodeOutput` (Advanced - see tutorial)
Node attributes assigned by the class. If unspecified, they are inferred from the `nodeAttributes` field.
This field is used together with the dependency fields of other rules/classes to know which rules/classes should be
updated when this class is assigned/removed to/from a node.

**Returns**
  *  `StyleClass`

## `ogma.styles.fontSizeToBadgeScale(fontSize, radius[, fontScale])`

Computes the right **fixed** badge scale for a given font size and node radius. Use it when you are combining the 'scaled' node size policy with 'fixed' badge size.

**Arguments**
  * `fontSize` `number` The desired font size at zoom 1
  * `radius` `number` The radius of the node
  * `fontScale`(optional) `number`[default: 0.5] The scale of the font, used to compute the badge size

**Returns**
  *  `number`

## `ogma.styles.getClass(className)`

Returns the class with the specified name. Returns `null` if no class has the specified name.

**Arguments**
  * `className` `string`

**Returns**
  *  `StyleClass|null`

## `ogma.styles.getClassList()`

Returns the list of existing classes by increasing priority, excluding builtin classes.

**Returns**
  *  `Array<StyleClass>`

## `ogma.styles.getEdgeRules()`

Returns all rules that only impact edges.

**Returns**
  *  `Array<StyleRule>`

## `ogma.styles.getNodeRules()`

Returns all rules that only impact nodes.

**Returns**
  *  `Array<StyleRule>`

## `ogma.styles.getRuleList()`

Returns the list of all rules, in the order they are applied.

**Returns**
  *  `Array<StyleRule>`

## `ogma.styles.setDisabledEdgeAttributes(attributes[, fullOverwrite])`

Set the style of edges when they are disabled. If `null` is specified, the default disabled style will be applied to disabled edges.

**Arguments**
  * `attributes` `EdgeAttributesValue|null` Attributes to apply to disabled edges
  * `fullOverwrite`(optional) `boolean`[default: false] If `false`, the specified attributes will be merged with the current attributes.
If `true`, the attributes applied when disabled will be exactly the ones supplied.

## `ogma.styles.setDisabledNodeAttributes(attributes[, fullOverwrite])`

Set the style of nodes when they are disabled. If `null` is specified, the default disabled style will be applied to disabled nodes.

**Arguments**
  * `attributes` `NodeAttributesValue|null` Attributes to apply to disabled nodes
  * `fullOverwrite`(optional) `boolean`[default: false] If `false`, the specified attributes will be merged with the current attributes.
If `true`, the attributes applied when disabled will be exactly the ones supplied.

## `ogma.styles.setEdgeTextsVisibility(value)`

Show or hide all the edge texts. This method has an internal counter; if it's called with `false`, the counter is
decreased by one, if it's called with `true` the counter is increased by one. The counter starts at 0, and
cannot go lower than 0.
The edge texts are shown if the counter is equal to 0.

**Arguments**
  * `value` `boolean`

## `ogma.styles.setEdgesVisibility(value)`

Show or hide all the edges. This method has an internal counter; if it's called with `false`, the counter is
decreased by one, if it's called with `true` the counter is increased by one. The counter starts at 0, and
cannot go lower than 0.
The edges are shown if the counter is equal to 0.

**Arguments**
  * `value` `boolean`

## `ogma.styles.setHoveredEdgeAttributes(attributes[, fullOverwrite])`

Set the style of edges when they are hovered. If `null` is specified, no style will be applied to hovered edges.

**Arguments**
  * `attributes` `HoverEdgeOptions|null` Attributes to apply to hovered edges
  * `fullOverwrite`(optional) `boolean`[default: false] If `false`, the specified attributes will be merged with the current attributes.
If `true`, the attributes applied on hover will be exactly the ones supplied.

## `ogma.styles.setHoveredEdgeExtremitiesAttributes(attributes[, fullOverwrite])`

Set the style of the extremities of the hovered edges. If `null` is specified, no style will be applied them.

**Arguments**
  * `attributes` `NodeAttributesValue|null` Attributes to apply to the extremities of hovered edges
  * `fullOverwrite`(optional) `boolean`[default: false] If `false`, the specified attributes will be merged with the current attributes.
If `true`, the attributes applied on hover will be exactly the ones supplied.

## `ogma.styles.setHoveredNodeAttributes(attributes[, fullOverwrite])`

Set the style of nodes when they are hovered. If `null` is specified, no style will be applied to hovered nodes.

**Arguments**
  * `attributes` `HoverNodeOptions|null` Attributes to apply to hovered nodes
  * `fullOverwrite`(optional) `boolean`[default: false] If `false`, the specified attributes will be merged with the current attributes.
If `true`, the attributes applied on hover will be exactly the ones supplied.

## `ogma.styles.setNodeTextsVisibility(value)`

Show or hide all the node texts. This method has an internal counter; if it's called with `false`, the counter is
decreased by one, if it's called with `true` the counter is increased by one. The counter starts at 0, and
cannot go lower than 0.
The node texts are shown if the counter is equal to 0.

**Arguments**
  * `value` `boolean`

## `ogma.styles.setNodesVisibility(value)`

Show or hide all the nodes. This method has an internal counter; if it's called with `false`, the counter is
decreased by one, if it's called with `true` the counter is increased by one. The counter starts at 0, and
cannot go lower than 0.
The nodes are shown if the counter is equal to 0.

**Arguments**
  * `value` `boolean`

## `ogma.styles.setSelectedEdgeAttributes(attributes[, fullOverwrite])`

Set the style of edges when they are selected. If `null` is specified, no style will be applied to selected edges.

**Arguments**
  * `attributes` `EdgeAttributesValue|null` Attributes to apply to selected edges
  * `fullOverwrite`(optional) `boolean`[default: false] If `false`, the specified attributes will be merged with the current attributes.
If `true`, the attributes applied on selection will be exactly the ones supplied.

## `ogma.styles.setSelectedEdgeExtremitiesAttributes(attributes[, fullOverwrite])`

Set the style of the extremities of the selected edges. If `null` is specified, no style will be applied them.

**Arguments**
  * `attributes` `NodeAttributesValue|null` Attributes to apply to the extremities of selected edges
  * `fullOverwrite`(optional) `boolean`[default: false] If `false`, the specified attributes will be merged with the current attributes.
If `true`, the attributes applied on selection will be exactly the ones supplied.

## `ogma.styles.setSelectedNodeAttributes(attributes[, fullOverwrite])`

Set the style of nodes when they are selected. If `null` is specified, no style will be applied to selected nodes.

**Arguments**
  * `attributes` `NodeAttributesValue|null` Attributes to apply to selected nodes
  * `fullOverwrite`(optional) `boolean`[default: false] If `false`, the specified attributes will be merged with the current attributes.
If `true`, the attributes applied on selection will be exactly the ones supplied.

## `ogma.styles.setTheme(theme)`

Sets the theme for Ogma

**Arguments**
  * `theme` `Theme`
      * `disabledEdgeAttributes`(optional) `EdgeAttributesValue` Disabled edge attributes
      * `disabledNodeAttributes`(optional) `NodeAttributesValue` Disabled node attributes
      * `edgeAttributes`(optional) `EdgeAttributes` Default edge attributes
      * `hoveredEdgeAttributes`(optional) `HoverEdgeOptions` Hovered edge attributes (you can specify duration and easing for the animation)
      * `hoveredNodeAttributes`(optional) `HoverNodeOptions` Hovered node attributes (you can specify duration and easing for the animation)
      * `nodeAttributes`(optional) `NodeAttributes` Default node attributes
      * `selectedEdgeAttributes`(optional) `EdgeAttributes` Selected edge attributes
      * `selectedNodeAttributes`(optional) `NodeAttributes` Selected node attributes

# Ogma.tools

  * Ogma.tools.brand

  * Ogma.tools.connectNodes

  * Ogma.tools.lasso

  * Ogma.tools.legend

  * Ogma.tools.rectangleSelect

  * Ogma.tools.resize

  * Ogma.tools.rewire

  * Ogma.tools.snapping

  * Ogma.tools.tooltip

# Ogma.tools.brand

Brand API: functions to display a brand component in the Ogma container.

## `ogma.tools.brand.remove()`

Remove the brand HTML element.

## `ogma.tools.brand.set(html[, options])`

Display the specified HTML in one of the corner of the canvas.

If the brand shows in one of the corner of the page instead of the Ogma container, change the css "position" attribute
of the container to "relative".

**Arguments**
  * `html` `string` HTML content to display.
  * `options`(optional) `BrandOptions`

**Returns**
  *  `HTMLElement`

# Ogma.tools.connectNodes

Connect Nodes API: functions to connect nodes by drawing edges.

## `ogma.tools.connectNodes.disable()`

Disable the "connectNodes" mode.

## `ogma.tools.connectNodes.enable([options])`

Enable the "connectNodes" mode, allowing the user to draw edges with the mouse.

**Arguments**
  * `options`(optional) `object`
      * `condition`(optional) `function(source: Node, target: Node): boolean` If specified, will only connect nodes that satisfy this condition.
      * `continueDrawing`(optional) `boolean`[default: false] If `true`, the mode will not be disabled after the first connection is created.
      * `createEdge`(optional) `function(rawEdge: RawEdge):RawEdge` Callback called before creating a new edge: allows you to change the id, data and attributes of the new edge.
      * `createNode`(optional) `function(rawNode: RawNode):RawNode` Callback called before creating a new node: allows you to change the id, data and attributes of the new node.
      * `createNodes`(optional) `boolean`[default: true] Indicates if a node should be created when pointing on an empty space.
      * `cursorStyle`(optional) `CursorStyle`[default: "cell"]
      * `dashLength`(optional) `number`[default: 8]
      * `onComplete`(optional) `function(source:Node, target: Node, edge: Edge):void` Called when a new connection is created. Note that target and edge could be `null` in case of no edge is created.
      * `onEdgeCreated`(optional) `function(edge: Edge):void` In this callback you can add properties or styles to the created edge.
      * `onNodeCreated`(optional) `function(node: Node):void` If `createNodes` is set to true, a new node will be created and here you can add properties or styles to it.
      * `strokeColor`(optional) `Color`[default: "black"]
      * `strokeWidth`(optional) `number`[default: 2]

## `ogma.tools.connectNodes.enabled()`

Indicates if the "connectNodes" mode is enabled.

# Ogma.tools.lasso

Lasso API: functions to select nodes and edges by drawing a lasso.

## `ogma.tools.lasso.disable()`

Disables lasso tool.

## `ogma.tools.lasso.enable([options])`

Enables the lasso selection tool

**Arguments**
  * `options`(optional) `object`
      * `bothExtremities`(optional) `boolean`[default: false] If set to `true`, edge will
be passed to `callback` only if both of its ends are inside the selected area.
By default, just one endpoint inside the selection is enough to be included.
      * `callback`(optional) `function(payload: {nodes: NodeList, edges: EdgeList, points: Point[] })` Called with the nodes and edges surrounded by the lasso.
By default, add the nodes to the selection (edges are ignored).
      * `cursorStyle`(optional) `CursorStyle`[default: "cell"] Cursor style when the lasso is active (CSS property)
      * `fillColor`(optional) `Color`[default: "rgba(0,195,255,0.1)"] Lasso fill color
      * `strokeColor`(optional) `Color`[default: "#00C3FF"] Lasso stroke color
      * `strokeWidth`(optional) `number`[default: 1] Lasso stroke width

## `ogma.tools.lasso.enabled()`

Check whether the lasso tool is enabled.

**Returns**
  *  `boolean`

# Ogma.tools.legend

Legend API: functions to display a legend in the Ogma container to explain the meaning of nodes and edges.

## `ogma.tools.legend.disable()`

Disable the legend.

**Returns**
  *  `Promise<void>`

## `ogma.tools.legend.enable([options])`

Enable the legend. Provides indications on the meaning of the color, size, shape, image and icon of nodes
and edges.

Important! In order to be generated for a specific attribute (color, size, etc), the following
requirements must be met:

  There must be exactly one rule for that attribute. If there are multiple rules for an attribute, only
  the first one is taken into account (which may lead to an incorrect legend)
  This rule must have been created with `ogma.rules.map()` or `ogma.rules.slice()`. If it's not the
  case, the widget for this attribute will not be shown.

**Arguments**
  * `options`(optional) `LegendOptions`

**Returns**
  *  `Promise<void>`

## `ogma.tools.legend.enabled()`

Indicates if the legend is enabled.

**Returns**
  *  `boolean`

## `ogma.tools.legend.export([settings])`

**Arguments**
  * `settings`(optional) `LegendOptions & Size`

**Returns**
  *  `Promise<HTMLCanvasElement>` Exports legend contents on a Canvas element

## `ogma.tools.legend.getOptions()`

Get Legend settings.

**Returns**
  *  `LegendOptions`

# Ogma.tools.rectangleSelect

Rectangle Select API: functions to select nodes and edges by drawing a rectangle.

## `ogma.tools.rectangleSelect.disable()`

Disable the rectangle selection.

## `ogma.tools.rectangleSelect.enable([options])`

Enable the selection by rectangle tool, allowing the user to select some nodes by drawing a rectangle on the screen.

**Arguments**
  * `options`(optional) `object`
      * `bothExtremities`(optional) `boolean`[default: false] If set to `true`, edge will
be passed to `callback` only if both of its ends are inside the selected area.
By default, just one endpoint inside the selection is enough to be included.
      * `callback`(optional) `function(evt: {nodes: NodeList, edges: EdgeList, rectangle: SimpleBoundingBox })` Called with the nodes/edges surrounded
by the rectangle. By default, add the surrounded nodes to the selection. If there is no surrounded node, add
the surrounded edges instead.
      * `cursorStyle`(optional) `CursorStyle`[default: "cell"] Cursor style when the rectangle is active (CSS property)
      * `fillColor`(optional) `Color|null`[default: "rgba(0,195,255,0.1)"] Rectangle fill color
      * `strokeColor`(optional) `Color`[default: "#00C3FF"] Rectangle stroke color
      * `strokeWidth`(optional) `number`[default: 1] Rectangle stroke width

## `ogma.tools.rectangleSelect.enabled()`

Indicates if the rectangle selection is enabled.

**Returns**
  *  `boolean`

# Ogma.tools.resize

Resize API: functions to resize nodes.

## `ogma.tools.resize.disable()`

Disable the "resize" mode.

## `ogma.tools.resize.enable([options])`

Enable the "resize" mode, allowing the user to manually change the size of the selected nodes.

**Arguments**
  * `options`(optional) `object`
      * `color`(optional) `Color`[default: "#00C3FF"] Color used to display the nodes bounding box
      * `cursor`(optional) `CursorStyle`[default: "nesw-resize"] CSS cursor to use when the mouse is over a resizing handle
      * `detectionMargin`(optional) `number`[default: 5] Maximum distance (in pixels) the mouse can be to the resize handler and still be detected
      * `handleSize`(optional) `number`[default: 6] Width (in pixels) of the square indicator to drag in order to resize the node
      * `lineWidth`(optional) `number`[default: 1] Width (in pixels) of the stroke of the square representing the nodes bounding box
      * `nbNodesToSnapTo`(optional) `number`[default: 5] Number of close nodes to use for snapping.
      * `previewColor`(optional) `Color`[default: &#039;rgba(0, 0, 0, 0.2)&#039;] Color of the preview of the node being resized
      * `sizeIndicatorColor`(optional) `Color`[default: "black"] Color of the size indicator (shown when the node snaps to the size of another node)
      * `sizeIndicatorOffset`(optional) `number`[default: 5] Offset (in pixels) to the left on which the size indicator must be displayed
      * `sizeIndicatorThickness`(optional) `number`[default: 1] Thickness (in pixels) of the line used to draw the size indicator
      * `sizeIndicatorWidth`(optional) `number`[default: 3] Total width (in pixels) of the indicator size
      * `snappingRatio`(optional) `number`[default: 1.25] Ratio used to determine the size to snap to, when the node does not snap to another node size. Must be strictly greater than 1.

## `ogma.tools.resize.enabled()`

Indicates if the "resize" mode is enabled.

**Returns**
  *  `boolean`

# Ogma.tools.rewire

Rewiring API: functions to manually change the extremities of the selected edges.

## `ogma.tools.rewire.disable()`

Disable the "rewire" tool'.

## `ogma.tools.rewire.enable([options])`

Enable the "rewire" tool, enabling the user to manually change the extremities of the selected edges.

**Arguments**
  * `options`(optional) `object`
      * `color`(optional) `Color`[default: "#00C3FF"] Color of the handle in the center of the nodes
      * `cursorOnDrag`(optional) `CursorStyle`[default: &#039;grabbing&#039;] CSS cursor style when dragging an edge
      * `cursorOnHover`(optional) `CursorStyle`[default: &#039;grab&#039;] CSS cursor style when hovering a node handle
      * `onMouseUpInVoid`(optional) `MouseUpInVoid` Callback function called when the handle is released in void (not on a node).
      * `radius`(optional) `number`[default: 7] Radius, in pixels, of the handle in the center of the nodes

## `ogma.tools.rewire.enabled()`

**Returns**
  *  `boolean` Indicates if the "rewire" tool is enabled

# Ogma.tools.snapping

Snapping API: functions to snap nodes to guidelines while dragging them.

## `ogma.tools.snapping.disable()`

Disables snapping mode in nodes dragging.

## `ogma.tools.snapping.enable([options])`

Enables the snapping mode while dragging the nodes.

**Arguments**
  * `options`(optional) `object`
      * `centerSnapDistance`(optional) `Number`[default: 240] Maximum distance between 2 nodes that allowes their centers to be aligned
      * `enabled`(optional) `Boolean`[default: false] Indicates whether the snapping is on when the node is being dragged
      * `guidelineColor`(optional) `Color`[default: &#039;red&#039;] Color of the axis-ligned guidelines
      * `guidelineWidth`(optional) `Number`[default: 1] Width of the alignment guideline
      * `neighbours`(optional) `object` Equal nodes distribution: snaps a node to the middle of the distance between two other nodes or to the same distance as two of its neighbours
          * `enabled`(optional) `Boolean`[default: true] Enables the mode.
          * `lineColor`(optional) `Color`[default: #00C3FF] Guideline color
          * `lineWidth`(optional) `Number`[default: 1] Guideline width
          * `offset`(optional) `Number`[default: 5] Distance between the guideline and common bounding box of aligned nodes
          * `tolerance`(optional) `Number`[default: 3] Snapping distance

      * `preferredDistance`(optional) `object` Options for preferred ditance snapping: pairwise snapping of the nodes based on their distance
          * `enabled`(optional) `Boolean`[default: true] Enables the mode.
          * `lineColor`(optional) `Color`[default: #00C3FF] Guideline color
          * `lineWidth`(optional) `Number`[default: 1] Guideline width
          * `ratio`(optional) `Number`[default: 1.13] Preferred distance ratio between nodes `a` and `b`, according to the formula `d = (Ra + Rb) * ratio`
          * `tolerance`(optional) `Number`[default: 10] Snapping distance

      * `sideSnapDistanceFactor`(optional) `Number`[default: 3] Maximum distance between 2 nodes that allowes their sides to be aligned. The value is the factor of dragged node diameter.
      * `tolerance`(optional) `Number`[default: 5] Pixel tolerance distance within which node is snapped towards the guideline

# Ogma.tools.tooltip

Tooltip API: functions to display tooltips when hovering or clicking on nodes and edges.

## `ogma.tools.tooltip.hide()`

Hide the tooltip that is currently being displayed.

## `ogma.tools.tooltip.isShown()`

Indicates if a tooltip is currently displayed.

**Returns**
  *  `boolean`

## `ogma.tools.tooltip.onBackgroundClick(handler[, options])`

Indicates the tooltip to display when the background is left clicked.

**Arguments**
  * `handler` `function (): (string|Promise<string>)` The function that will be called to generate the tooltip. Must return a HTML string.
  * `options`(optional) `TooltipOptions`

## `ogma.tools.tooltip.onBackgroundDoubleClick(handler[, options])`

Indicates the tooltip to display when the background is double clicked.

**Arguments**
  * `handler` `function (): (string|Promise<string>)` The function that will be called to generate the tooltip. Must return a HTML string.
  * `options`(optional) `TooltipOptions`

## `ogma.tools.tooltip.onBackgroundRightClick(handler[, options])`

Indicates the tooltip to display when the background is right clicked.

**Arguments**
  * `handler` `function (): (string|Promise<string>)` The function that will be called to generate the tooltip. Must return a HTML string.
  * `options`(optional) `TooltipOptions`

## `ogma.tools.tooltip.onEdgeClick(handler[, options])`

Indicates the tooltip to display when a edge is left clicked.

**Arguments**
  * `handler` `function (edge: Edge): (string|Promise<string>)` The function that will be called to generate the tooltip. Must return a HTML string.
  * `options`(optional) `TooltipOptions`

## `ogma.tools.tooltip.onEdgeDoubleClick(handler[, options])`

Indicates the tooltip to display when a edge is double clicked.

**Arguments**
  * `handler` `function (edge: Edge): (string|Promise<string>)` The function that will be called to generate the tooltip. Must return a HTML string.
  * `options`(optional) `TooltipOptions`

## `ogma.tools.tooltip.onEdgeHover(handler[, options])`

Indicates the tooltip to display when a edge is hovered.

**Arguments**
  * `handler` `function (edge: Edge): (string|Promise<string>)` The function that will be called to generate the tooltip. Must return a HTML string.
  * `options`(optional) `TooltipOptions`

## `ogma.tools.tooltip.onEdgeRightClick(handler[, options])`

Indicates the tooltip to display when a edge is right clicked.

**Arguments**
  * `handler` `function (edge: Edge): (string|Promise<string>)` The function that will be called to generate the tooltip. Must return a HTML string.
  * `options`(optional) `TooltipOptions`

## `ogma.tools.tooltip.onNodeClick(handler[, options])`

Indicates the tooltip to display when a node is left clicked.

**Arguments**
  * `handler` `function (node: Node): (string|Promise<string>)` The function that will be called to generate the tooltip. Must return a HTML string.
  * `options`(optional) `TooltipOptions`

## `ogma.tools.tooltip.onNodeDoubleClick(handler[, options])`

Indicates the tooltip to display when a node is double clicked.

**Arguments**
  * `handler` `function (node: Node): (string|Promise<string>)` The function that will be called to generate the tooltip. Must return a HTML string.
  * `options`(optional) `TooltipOptions`

## `ogma.tools.tooltip.onNodeHover(handler[, options])`

Indicates the tooltip to display when a node is hovered.

**Arguments**
  * `handler` `function (node: Node): (string|Promise<string>)` The function that will be called to generate the tooltip. Must return a HTML string.
  * `options`(optional) `TooltipOptions`

## `ogma.tools.tooltip.onNodeRightClick(handler[, options])`

Indicates the tooltip to display when a node is right clicked.

**Arguments**
  * `handler` `function (node: Node): (string|Promise<string>)` The function that will be called to generate the tooltip. Must return a HTML string.
  * `options`(optional) `TooltipOptions`

## `ogma.tools.tooltip.refresh()`

Refresh the current tooltip.

# Ogma.transformations

Transformations allow to change the structure of the graph based on rules.

## `ogma.transformations.addDrillDown([options])`

Add the ability to replace a node by a group of nodes and edges.
For instance, when double click on a node, you can can call drilldown.drill(clickedNode)
to replace the clickedNode by its children.

**Arguments**
  * `options`(optional) `DrillDownOptions`

**Returns**
  *  `DrillDown` The drilldown

## `ogma.transformations.addEdgeFilter(options)`

Hide all edges for which the specified function evaluate to false. The filter is applied as long as it's
not removed and is updated when edges are added/removed/their data is updated. Also hides the edges adjacent
to the hidden edges.

**Arguments**
  * `options` `EdgeFilterOptions|function(edge: Edge): boolean`

**Returns**
  *  `Transformation`

## `ogma.transformations.addEdgeGrouping([options])`

Group the edges that are parallel and match the same group id together. The groups are automatically updated when the original edges
are added, removed or when their data is updated.

**Arguments**
  * `options`(optional) `EdgeGroupingOptions`

**Returns**
  *  `Transformation` The added transformation

## `ogma.transformations.addGeoClustering([options])`

Group nodes in geo mode. Works in a similar  way as **addNodeGrouping** but only in geo mode.
The nodes get grouped depending on the distance between each other in Mercator projection.
**Warning**: you should have only one GeoClustering object in your vizualisation to prevent from conflicts.

**Arguments**
  * `options`(optional) `GeoClusteringOptions`

**Returns**
  *  `Transformation` The added transformation

## `ogma.transformations.addNeighborGeneration(options)`

For each node that match the selector, create one or more neighbor nodes, each identified by a string.
If multiple nodes give the same string identifier, they will all share the same neighbor.

**Arguments**
  * `options` `NeighborGenerationOptions`

**Returns**
  *  `Transformation`

## `ogma.transformations.addNeighborMerging(options)`

Hide the nodes that match the selector, and add the data properties
specified by `dataFunction` to their neighbors. When the transformation is
destroyed or disabled, the data of the affected nodes is restored.

**Arguments**
  * `options` `NeighborMergingOptions`

**Returns**
  *  `Transformation`

## `ogma.transformations.addNodeClustering([options])`

Middleware on `addNode` `addEdge` which allows to cluster the graph without adding the subNodes and subEdges to it.
This is much faster than Grouping as the graph remains small. Switching between clustered/unclustered
state is though slower than with the Grouping (as Ogma has to recreate all the missing nodes and edges).

**Arguments**
  * `options`(optional) `NodeClusteringOptions`

**Returns**
  *  `Transformation` The added transformation

## `ogma.transformations.addNodeCollapsing(options)`

Hide the nodes that match the specified selector. For each of the hidden nodes, create an edge between each
pair of adjacent nodes of this node. In essence, it's transforming the graph such as (A -> B -> C) becomes
(A -> C), with B being a node that matches the selector.
Note that this will throw in the case of multiple collapsed nodes, for instance a daisy chain of nodes like (A -> B -> C -> D)
where B and C are collapsed.

**Arguments**
  * `options` `NodeCollapsingOptions`

**Returns**
  *  `Transformation`

## `ogma.transformations.addNodeFilter(options)`

Hide all nodes for which the specified function evaluate to false. The filter is applied as long as it's
not removed and is updated when nodes are added/removed/their data is updated. Also hides the edges adjacent
to the hidden nodes.

**Arguments**
  * `options` `NodeFilterOptions|function(node: Node): boolean`

**Returns**
  *  `Transformation`

## `ogma.transformations.addNodeGrouping([options])`

Group the nodes that match the same group id together. The groups are automatically updated when original nodes
are added, removed or when their data is updated.

**Arguments**
  * `options`(optional) `NodeGroupingOptions`

**Returns**
  *  `Transformation` The added transformation

## `ogma.transformations.addVirtualProperties(options)`

Add (or overwrite) some data properties to the specified nodes and edges. When the transformation
is disabled/destroyed, the old data is restored.

**Arguments**
  * `options` `VirtualPropertiesOptions`

**Returns**
  *  `Transformation`

## `ogma.transformations.afterNextUpdate()`

Returns a Promise that resolves after the next time the transformations are updated

**Returns**
  *  `Promise<void>`

## `ogma.transformations.clear()`

Clear all transformations.

**Returns**
  *  `Promise<void>`

## `ogma.transformations.collapseGroup(options)`

Collapse a virtual node, its radius is updated and its neighbors are
repositioned following the fisheye algorithm. The subnodes are positioned
to the center of the focus node.

**Arguments**
  * `options` `object`
      * `node` `Node` Node to collapse.
      * `onCollapse` `function(): Promise<void>` Callback tagging the focus node as closed.

**Returns**
  *  `Promise<void>` The promise resolves when the collapse is done.

## `ogma.transformations.expandGroup(options)`

Expand a virtual node, its radius is updated and its neighbors are
repositioned following the fisheye algorithm. The subnodes are positioned
within the focus node according to the `computeSubNodesPosition` function.

**Arguments**
  * `options` `object`
      * `computeSubNodesPosition` `function(nodes: NodeList): Promise<Point[]>` Function to compute the position of subnodes within the focus node.
      * `node` `Node` Node to expand.

**Returns**
  *  `Promise<void>` The promise resolves when the expansion is done.

## `ogma.transformations.getById(id)`

**Arguments**
  * `id` `Number`

**Returns**
  *  `Transformation | undefined`

## `ogma.transformations.getList()`

Returns the list of transformations applied to the graph.

**Returns**
  *  `Array<Transformation>`

## `ogma.transformations.getXYR(nodes)`

Returns the x, y and radius values of the nodes during transformations update.
This method is useful to create your own custom layout with multilevel open grouping.

**Arguments**
  * `nodes` `NodeList`

**Returns**
  *  `Transformation`

## `ogma.transformations.layoutGroups(toRelayout, options, propagate)`

Run layouts on the groups that are specified in the arguments.

**Arguments**
  * `toRelayout` `GroupLayout` Array of objects containing the groupId, the layout name and the params of the layout to run
  * `options` `AttributeAnimationOptions` Animation options (the ones passed into toRelayout are ignored)
  * `propagate` `boolean` If true, re-layout the parent nodes of the groups, if false, simply recompute the radius and position of the parents.

**Returns**
  *  `Promise<void>` The promise resolves when the layouts are done.

## `ogma.transformations.onGroupsUpdated(callback)`

Defines the callback to be called when the node groups are updated.
Usually a layout.
- topLevelNodes: The nodes and groups that are not part of any group.
- radiuses: The radiuses of the top level nodes.

**Arguments**
  * `callback` `function(topLevelNodes: NodeList, radiuses: number[]): SubLayout`

## `ogma.transformations.triggerGroupUpdated(groups)`

Tells the transformations module to rerun the `onGroupUpdated` callback on next transformation update
on each group passed as argument
By default the layouts are run only if a node/edge is added/removed from a group

**Arguments**
  * `groups` `NodeList` The groups to update

## `ogma.transformations.triggerGroupsUpdated()`

Tells the transformations module to rerun the `onGroupsUpdated` callback on next transformation update
By default the layouts are run only if a node/edge is added/removed from a group

# Ogma.view

View API: functions to manipulate the view of the graph.

## `ogma.view.afterNextFrame()`

Returns a Promise that resolves after the next frame is rendered.

**Returns**
  *  `Promise<void>`

## `ogma.view.animationInProgress()`

Checks if any camera movement animations are currently in progress and
returns `true` if there are.

**Returns**
  *  `boolean`

## `ogma.view.beforeNextFrame()`

Returns a Promise that resolves before the next frame is rendered.

**Returns**
  *  `Promise<void>`

## `ogma.view.forceResize()`

Forces the canvas to be re-sized according to the container. Typically
useful when the visibility of the Ogma container changes, to notify Ogma
that it must refresh the scene.

## `ogma.view.get()`

Retrieve the current view.

**Returns**
  *  `{x: number, y: number, zoom: number, angle: number, width: number, height: number}`

## `ogma.view.getAngle()`

Indicates the current angle of the view.

**Returns**
  *  `number` Angle, in radians.

## `ogma.view.getBounds()`

Returns the bounds of the current view.

**Returns**
  *  `SimpleBoundingBox`

## `ogma.view.getCenter()`

Indicates the center of the view.

**Returns**
  *  `{x: number, y: number}`

## `ogma.view.getElementAt(pos)`

Returns the element located at the specified screen coordinates.

**Arguments**
  * `pos` `object`
      * `x` `number` Screen x
      * `y` `number` Screen y

**Returns**
  *  `Node|Edge|null`

## `ogma.view.getElementsInView()`

Returns the element currently visible in the viewport

**Returns**
  *  `{ nodes: NodeList, edges: EdgeList }`

## `ogma.view.getElementsInside(xmin, ymin, xmax, ymax, convertToScreenCoordinates)`

Returns elements inside of the rectangle defined by the screen coordinates.

**Arguments**
  * `xmin` `number` X coordinate of the bottom left corner
  * `ymin` `number` Y coordinate of the bottom left corner
  * `xmax` `number` X coordinate of the top right corner
  * `ymax` `number` Y coordinate of the top right corner
  * `convertToScreenCoordinates` `boolean = true` Whether or not the function will convert the coordinates
to screen space before running the query

**Returns**
  *  `{ nodes: NodeList, edges: EdgeList }`

## `ogma.view.getGraphBoundingBox([options])`

Returns the bounding box of the graph, in graph coordinates.

**Arguments**
  * `options`(optional) `object`
      * `includeTexts`(optional) `boolean`[default: false] Wether or not take texts in account in the bouding box

**Returns**
  *  `BoundingBox`

## `ogma.view.getImageData()`

Returns a new `ImageData`, containing the pixels
that are displayed by the current renderer. If the current renderer is not canvas or WebGL, the method returns
`null`.

If the screen has a pixel density greater than one (for example, retina screen), the retrieved ImageData will
not be rescaled and will have a bigger width and height than the view.

Note: since it's simply copying pixels, this method is way faster than an image export
(`ogma.export.png()` for example).

**Returns**
  *  `Promise<ImageData|null>`

## `ogma.view.getNodeBadgeAt(node, point)`

Returns which badge is under the specified point. Useful to know if a node badge is under the mouse cursor.

**Arguments**
  * `node` `Node|Edge` The node to check.
  * `point` `Point` The point to check.

**Returns**
  *  `topLeft|topRight|bottomLeft|bottomRight|undefined` The badge under the point, or undefined if none.

## `ogma.view.getSize()`

Returns the view width and height.

**Returns**
  *  `{ width: number, height: number }`

## `ogma.view.getTextBoundingBox(item)`

Get screen coordinates of the axis-aligned bounding box of the specified node or edge.

**Arguments**
  * `item` `Node|Edge`

**Returns**
  *  `BoundingBox|null`

## `ogma.view.getZoom()`

Indicates the current zoom level.

**Returns**
  *  `number`

## `ogma.view.graphToScreenCoordinates(coordinates)`

Returns a position on the screen from graph coordinates (e.g a node's position).

**Arguments**
  * `coordinates` `{x: number, y: number}`

**Returns**
  *  `{x: number, y: number}`

## `ogma.view.isFullScreen()`

Indicates if the full screen mode is currently enabled.

**Returns**
  *  `boolean`

## `ogma.view.locateGraph([options])`

Centers the view on the graph.

**Arguments**
  * `options`(optional) `LocateOptions`

**Returns**
  *  `Promise<void>`

## `ogma.view.locateRawGraph(graph[, options])`

Centers the view on the specified raw graph.

**Arguments**
  * `graph` `RawGraph`
  * `options`(optional) `LocateOptions`

**Returns**
  *  `Promise<void>`

## `ogma.view.move(offset[, options])`

Move the center of the view by the specified amount of pixels (NOT graph coordinates).

**Arguments**
  * `offset` `{x: number, y: number}`
  * `options`(optional) `CameraAnimationOptions`

**Returns**
  *  `Promise<void>`

## `ogma.view.moveTo(view[, options])`

Smoothly animate to the particular view. If you don't provide the
`duration` option, the optimal animation duration is used based on the
translation distance.

**Arguments**
  * `view` `object`
      * `x` `number` X coordinate of the center of the view.
      * `y` `number` Y coordinate of the center of the view.
      * `zoom` `number` Zoom level of the view.

  * `options`(optional) `CameraAnimationOptions` For duration and easing

**Returns**
  *  `Promise<void>`

## `ogma.view.moveToBounds(target[, options])`

Animates the view to the bounding box. If you don't provide the
`duration` option, the optimal animation duration is used based on the
translation distance.

**Arguments**
  * `target` `BoundingBox | [number, number, number, number]` It can be a `BoundingBox` or an Array of `minX`, `minY`, `maxX`, `maxY` in graph coordinates
  * `options`(optional) `CameraAnimationOptions` For duration and easing

**Returns**
  *  `Promise<void>`

## `ogma.view.rotate(angle[, options])`

Rotate the view by the specified angle.

**Arguments**
  * `angle` `number` Angle, in radians.
  * `options`(optional) `CameraAnimationOptions`

**Returns**
  *  `Promise<void>`

## `ogma.view.screenToGraphCoordinates(coordinates)`

Returns graph coordinates from a position on the screen.

**Arguments**
  * `coordinates` `{x: number, y: number}`

**Returns**
  *  `{x: number, y: number}`

## `ogma.view.set(view[, options])`

Set the view.

**Arguments**
  * `view` `object`
      * `angle`(optional) `number` Angle of the view, in radians.
      * `x`(optional) `number` X coordinate of the center of the view.
      * `y`(optional) `number` Y coordinate of the center of the view.
      * `zoom`(optional) `number` Zoom level of the view.

  * `options`(optional) `CameraAnimationOptions`

**Returns**
  *  `Promise<void>`

## `ogma.view.setAngle(angle[, options])`

Set the angle of the view.

**Arguments**
  * `angle` `number` Angle, in radians.
  * `options`(optional) `CameraAnimationOptions`

**Returns**
  *  `Promise<void>`

## `ogma.view.setCenter(center[, options])`

Set the center of the view, in graph coordinates.

**Arguments**
  * `center` `{x: number, y: number}`
  * `options`(optional) `CameraAnimationOptions`

**Returns**
  *  `Promise<void>`

## `ogma.view.setFullScreen(value)`

Enable or disable the full screen mode.

**Arguments**
  * `value` `boolean` `true` to enable the full screen mode, `false` to disable it.

**Returns**
  *  `Promise<void>`

## `ogma.view.setSize(size)`

Set the size of the view. This method only has effect if this instance of
Ogma has no container.

**Arguments**
  * `size` `object`
      * `height` `number`
      * `width` `number`

**Returns**
  *  `Promise<void>`

## `ogma.view.setZoom(zoom[, options])`

Set the zoom level.

**Arguments**
  * `zoom` `number`
  * `options`(optional) `CameraAnimationOptions`

**Returns**
  *  `Promise<void>`

## `ogma.view.zoomIn([options])`

Multiply the current zoom level by the specified amount.
If geo mode is on, it will fall back to `ogma.geo.setZoom` and use the zoom modifier defined by the projection.

**Arguments**
  * `options`(optional) `number|ZoomAnimationOptions` Zoom modifier or options. If `modifier` is not specified, it will be set to `Math.sqrt(2)`.

**Returns**
  *  `Promise<void>`

## `ogma.view.zoomOut([options])`

Divide the current zoom level by the specified amount.
If geo mode is on, it will fall back to `ogma.geo.setZoom` and use the zoom modifier defined by the projection.

**Arguments**
  * `options`(optional) `number|ZoomAnimationOptions` Zoom modifier or options. If `modifier` is not specified, it will be set to `Math.sqrt(2)`.

**Returns**
  *  `Promise<void>`

# Ogma.geometry

  * Ogma.geometry.lines

# Ogma.geometry.lines

Ogma static toolbox for dealing with line geometries

## `ogma.geometry.lines.isPointInLine(px, py, x1, y1, x2, y2, width, margin, isTriangle)`

Checks if a point (px, py) is inside a line ((x1, y1), (x2, y2)).

**Arguments**
  * `px` `number`
  * `py` `number`
  * `x1` `number`
  * `y1` `number`
  * `x2` `number`
  * `y2` `number`
  * `width` `number` Width of the line
  * `margin` `number`
  * `isTriangle` `boolean`

**Returns**
  *  `boolean`

# Edge

Ogma edge entity class. Use this class to manipulate edges, change data, attributes, etc.
 * ## `edge.isNode`

Read-only property that is always `false`.

## `edge.addClass(className[, options])`

Add the specified class to the edge.

**Arguments**
  * `className` `string`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Edge>`

## `edge.addClasses(classNames[, options])`

Add the specified classes to the edge.

**Arguments**
  * `classNames` `Array<string>`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Edge>`

## `edge.getAdjacentElements()`

Retrieves the list of edges parallel to the edge, excluding the source edge plus the extremities of the edge.

**Returns**
  *  `SubGraph`

## `edge.getAttribute(attributeName)`

Returns the value of the specified attribute for the edge.

**Arguments**
  * `attributeName` `PropertyPath` Attribute to retrieve.

**Returns**
  *  `any`

## `edge.getAttributes([attributeNames])`

Returns an object containing the specified attributes for the edge.

**Arguments**
  * `attributeNames`(optional) `Array<PropertyPath>` List of attributes to include in the object. If not specified, includes all the edge attributes.

**Returns**
  *  `EdgeAttributes`

## `edge.getBoundingBox([options])`

Returns the bounding box of the edge, in graph coordinates.

**Arguments**
  * `options`(optional) `object`
      * `ignoreCurvature`(optional) `boolean`[default: false] Use it if you want to only take into
                                         account the edge sources and targets.
      * `includeTexts`(optional) `boolean`[default: false] Wether or not take texts in account in the bouding box

**Returns**
  *  `BoundingBox`

## `edge.getClassList()`

Returns the list of classes that the edge has.

**Returns**
  *  `Array<string>`

## `edge.getData([property])`

Retrieve the specified data property of the edge. If no property is specified, retrieve the whole data object.
This method method returns the internal data object; modifying it could cause unexpected behavior.

**Arguments**
  * `property`(optional) `PropertyPath`

**Returns**
  *  `any`

## `edge.getExtremities()`

Returns a `NodeList` containing the source and the target of the edge.

**Returns**
  *  `NodeList`

## `edge.getId()`

Returns the id of the edge.

**Returns**
  *  `EdgeId`

## `edge.getMetaEdge()`

If the edge is grouped inside a meta-edge, returns this meta-edge. Otherwise, returns null.

**Returns**
  *  `Edge|null`

## `edge.getParallelEdges([options])`

Retrieves the list of edges parallel to the edge, including the source edge itself.

**Arguments**
  * `options`(optional) `object`
      * `filter`(optional) `Filter`[default: "visible"] Indicates which edges to take into account

**Returns**
  *  `EdgeList`

## `edge.getSource()`

Returns the source node of the edge

**Returns**
  *  `Node`

## `edge.getSubEdges()`

If the edge is a meta-edge (result of a grouping), returns the list of edges that are part of the group
it represents. If it's not a meta-edge, returns `null`.

**Returns**
  *  `EdgeList|null`

## `edge.getTarget()`

Returns the target node of the edge

**Returns**
  *  `Node`

## `edge.getTransformation()`

Returns the transformation that created the edge, if it is virtual. Otherwise returns `null`.

**Returns**
  *  `Transformation|null`

## `edge.hasClass(className)`

Indicates if the edge has the specified class.

**Arguments**
  * `className` `string`

**Returns**
  *  `boolean`

## `edge.isDisabled()`

Indicates if the edge is currently disabled.

**Returns**
  *  `boolean`

## `edge.isInView([options])`

Indicates if the edge is visible in the current view.

**Arguments**
  * `options`(optional) `object`
      * `margin`(optional) `number`[default: 0] Tolerance in pixels.

**Returns**
  *  `boolean`

## `edge.isSelected()`

Indicates if the edge is currently selected.

**Returns**
  *  `boolean`

## `edge.isVirtual()`

Indicates if the edge was created by a transformation (`true`) or not (`false`).

**Returns**
  *  `boolean`

## `edge.isVisible()`

Indicates if the edge is visible. A edge is not visible if it has been filtered out, or if it is used in a
transformation.
/!\ A edge with an opacity of 0 is considered visible!

**Returns**
  *  `boolean`

## `edge.locate([options])`

Centers the view on the edge.

**Arguments**
  * `options`(optional) `LocateOptions`

**Returns**
  *  `Promise<void>`

## `edge.pulse([options])`

Highlights the edge. It's a shorthand for the case when you
want the elements pulse for `number * (interval - 1) + duration` milliseconds.
It will also update the pulse attributes of the items with the one provided
in the `.pulse()` call.

**Arguments**
  * `options`(optional) `object`
      * `duration`(optional) `number`[default: 1000] Duration of a pulse (milliseconds)
      * `endColor`(optional) `Color|"inherit"`[default: "rgb(0,0,0,0.0)"] Ending color of the pulse
      * `endRatio`(optional) `number`[default: 2] Where the pulse ends, relative to the edge siz (1 = at the edge's border)
      * `interval`(optional) `number`[default: 800] Interval between two pulses (milliseconds)
      * `number`(optional) `number`[default: 1] Number of pulses
      * `startColor`(optional) `Color|"inherit"`[default: "rgb(0,0,0,0.6)"] Starting color of the pulse
      * `startRatio`(optional) `number`[default: 1] Where the pulse starts, relative to the edge siz (1 = at the edge's border)
      * `width`(optional) `number`[default: 10] Width of the pulse in pixels

**Returns**
  *  `Promise<Edge>`

## `edge.removeClass(className[, options])`

Remove the specified class from the edge.

**Arguments**
  * `className` `string`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Edge>`

## `edge.removeClasses(classNames[, options])`

Remove the specified class from the edge.

**Arguments**
  * `classNames` `Array<string>`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Edge>`

## `edge.resetAttributes([attributeNames][, options])`

Remove all attributes that have been applied through `setAttributes`.
Original attributes or attributes applied by the rules are not affected.

**Arguments**
  * `attributeNames`(optional) `Array<PropertyPath>` List of attributes to clear.
If no attribute is specified, clear all of them.
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Edge>`

## `edge.setAttribute(attribute, value[, options])`

Set the specified attribute of the edge.

**Arguments**
  * `attribute` `PropertyPath`
  * `value` `any`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Edge>`

## `edge.setAttributes(attributes[, options])`

Set the individual attributes of the edge.

**Arguments**
  * `attributes` `EdgeAttributesValue` Attributes to update
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Edge>`

## `edge.setData([property], value)`

Set the specified data property of the edge. If no property is specified, update the whole data object.

**Arguments**
  * `property`(optional) `PropertyPath`
  * `value` `any|function(edge: Edge): any`

**Returns**
  *  `Edge`

## `edge.setDisabled(active)`

Add or remove the edge to/from the disabled state.

**Arguments**
  * `active` `boolean` Whether to disable or enable the edge.

## `edge.setSelected(active)`

Add or remove the edge to/from the selection.

**Arguments**
  * `active` `boolean` Whether to select or unselect the edge.

## `edge.setSource(source)`

Set the source node of the edge.

**Arguments**
  * `source` `Node|NodeId`

## `edge.setTarget(target)`

Set the target node of the edge.

**Arguments**
  * `target` `Node|NodeId`

## `edge.setVisible(value)`

Hide or show the edge.

**Arguments**
  * `value` `boolean` Whether to show or hide the edge.

## `edge.toJSON([options])`

Returns an object containing the id, source id, target id, attributes and data of the edge.

**Arguments**
  * `options`(optional) `object`
      * `attributes`(optional) `Array<PropertyPath>|"all"`[=[]] List of attributes to retrieve. By default, retrieve all attributes.
      * `data`(optional) `function (data: any): any` Function that takes the edge's data in input and return
the data to retrieve. By default return the whole object.

**Returns**
  *  `RawEdge`

## `edge.toList()`

Returns a new EdgeList that contains only the edge.

**Returns**
  *  `EdgeList`

# EdgeList

Ogma edge list class. Use this class to manipulate a list of edges, change data, attributes, etc.
 * ## `edgeList.isNode`

Read-only property that is always `false`.
 * ## `edgeList.size`

Read-only property that indicates the number of edges in the list.

## `edgeList.addClass(className[, options])`

Add the specified class to the edges.

**Arguments**
  * `className` `string`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<EdgeList>`

## `edgeList.addClasses(classNames[, options])`

Add the specified classes to the edges.

**Arguments**
  * `classNames` `Array<string>`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<EdgeList>`

## `edgeList.concat(edges)`

**Arguments**
  * `edges` `EdgeList`

**Returns**
  *  `EdgeList`

## `edgeList.dedupe()`

Returns a new EdgeList which does not contain any duplicate edge.

**Returns**
  *  `EdgeList`

## `edgeList.every(callback)`

**Arguments**
  * `callback` `function(edge: Edge, index: number, list: EdgeList): boolean`

**Returns**
  *  `boolean`

## `edgeList.fillData([property], value)`

Set the specified data property of the edges with the same value.

**Arguments**
  * `property`(optional) `PropertyPath|any` Path of the data property to update. If no property is specified, update the whole data object.
  * `value` `any` Value that will be assigned to all the edges.

**Returns**
  *  `EdgeList`

## `edgeList.filter(callback)`

**Arguments**
  * `callback` `function(edge: Edge, index: number, list: EdgeList): boolean`

**Returns**
  *  `EdgeList`

## `edgeList.find(callback)`

**Arguments**
  * `callback` `function(edge: Edge, index: number, list: EdgeList): boolean`

**Returns**
  *  `Edge`

## `edgeList.forEach(callback)`

**Arguments**
  * `callback` `function(edge: Edge, index: number, list: EdgeList)`

## `edgeList.get(index)`

Returns the edge at the specified index.

**Arguments**
  * `index` `number`

**Returns**
  *  `Edge`

## `edgeList.getAdjacentElements()`

Retrieves the list of edges parallel to the edges, excluding the source edges themselves plus the extremities of the edges.

**Returns**
  *  `EdgeList`

## `edgeList.getAttribute(attributeName)`

Returns an array containing the value of the specified attribute for each edge.

**Arguments**
  * `attributeName` `PropertyPath` Attribute to retrieve.

**Returns**
  *  `any[]`

## `edgeList.getAttributes([attributes])`

Returns an array of objects containing the specified attributes for each edge.

**Arguments**
  * `attributes`(optional) `PropertyPath[]` List of attributes to include in the object. If not specified, includes all the edge attributes.

**Returns**
  *  `EdgeAttributes[]`

## `edgeList.getBoundingBox([options])`

Returns the bounding box of the edges, in graph coordinates.

**Arguments**
  * `options`(optional) `object`
      * `ignoreCurvature`(optional) `boolean`[default: false] Use it if you want to only take into
                                         account the edge sources and targets.
      * `includeTexts`(optional) `boolean`[default: false] Wether or not take texts in account in the bouding box

**Returns**
  *  `BoundingBox`

## `edgeList.getClassList()`

Returns the list of classes that each edge has.

**Returns**
  *  `Array<Array<string>>`

## `edgeList.getData([property])`

Retrieve the specified data property. If no property is specified, retrieve the whole data object.
This method method returns the internal data object; modifying it could cause unexpected behavior.

**Arguments**
  * `property`(optional) `PropertyPath`

**Returns**
  *  `Array<any>`

## `edgeList.getExtremities()`

Returns a `NodeList` containing the sources and targets of the edges. Duplicate nodes are not removed.

**Returns**
  *  `NodeList`

## `edgeList.getId()`

Returns the id of each edge.

**Returns**
  *  `Array<EdgeId>`

## `edgeList.getMetaEdge()`

Run `getMetaEdge` on each edge in the list and returns the array of results.

**Returns**
  *  `Array<Edge|null>`

## `edgeList.getParallelEdges([options])`

Retrieves the list of edges parallel to the edges, including the source edges themselves.

**Arguments**
  * `options`(optional) `object`
      * `filter`(optional) `Filter`[default: "visible"] Indicates which edges to take into account

**Returns**
  *  `EdgeList`

## `edgeList.getSource()`

Returns the list of source nodes of the edges

**Returns**
  *  `NodeList`

## `edgeList.getSubEdges()`

Run `getSubEdges` on all the edges in the list and returns the array of results

**Returns**
  *  `Array<EdgeList|null>`

## `edgeList.getTarget()`

Returns the list of target nodes of the edges

**Returns**
  *  `NodeList`

## `edgeList.includes(edge)`

Indicates if the `EdgeList` contains the specified edge.

**Arguments**
  * `edge` `Edge`

**Returns**
  *  `boolean`

## `edgeList.inverse()`

Returns a new EdgeList containing all the visible edges that are not in the list.

**Returns**
  *  `EdgeList`

## `edgeList.isDisabled()`

Indicates for each edge if it is disabled.

**Returns**
  *  `Array<boolean>`

## `edgeList.isSelected()`

Indicates if the edges are currently selected.

**Returns**
  *  `Array<boolean>`

## `edgeList.isVisible()`

Call `isVisible` on each edge in the list, and returns the array of results.

**Returns**
  *  `Array<boolean>`

## `edgeList.locate([options])`

Centers the view on the edges.

**Arguments**
  * `options`(optional) `LocateOptions`

**Returns**
  *  `Promise<void>`

## `edgeList.map(callback)`

**Arguments**
  * `callback` `function(edge: Edge, index: number, list: EdgeList): any`

**Returns**
  *  `Array<any>`

## `edgeList.pulse([options])`

Highlights the edges. It's a shorthand for the case when you
want the elements pulse for `number * (interval - 1) + duration` milliseconds.
It will also update the pulse attributes of the items with the one provided
in the `.pulse()` call

**Arguments**
  * `options`(optional) `object`
      * `duration`(optional) `number`[default: 1000] Duration of a pulse (milliseconds)
      * `endColor`(optional) `Color|"inherit"`[default: "rgb(0,0,0,0.0)"] Ending color of the pulse
      * `endRatio`(optional) `number`[default: 2] Where the pulse ends, relative to the edge siz (1 = at the edge's border)
      * `interval`(optional) `number`[default: 800] Interval between two pulses (milliseconds)
      * `number`(optional) `number`[default: 1] Number of pulses
      * `startColor`(optional) `Color|"inherit"`[default: "rgb(0,0,0,0.6)"] Starting color of the pulse
      * `startRatio`(optional) `number`[default: 1] Where the pulse starts, relative to the edge siz (1 = at the edge's border)
      * `width`(optional) `number`[default: 10] Width of the pulse in pixels

**Returns**
  *  `Promise<EdgeList>`

## `edgeList.reduce(callback, initialValue)`

**Arguments**
  * `callback` `function(accumulator: any, currentValue: Edge, index: number): any`
  * `initialValue` `any`

**Returns**
  *  `any`

## `edgeList.removeClass(className[, options])`

Remove the specified class from the edges.

**Arguments**
  * `className` `string`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<EdgeList>`

## `edgeList.removeClasses(classNames[, options])`

Remove the specified class from the edges.

**Arguments**
  * `classNames` `Array<string>`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<EdgeList>`

## `edgeList.resetAttributes([attributes][, options])`

Remove all attributes that have been applied through `setAttributes` of all the edges in the list.
Original attributes or attributes applied by the rules are not affected.

**Arguments**
  * `attributes`(optional) `Array<PropertyPath>` List of attributes to clear. If no attribute is specified, clear all of them.
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<EdgeList>`

## `edgeList.setAttribute(attribute, value[, options])`

Set the specified attribute of all the edges in the list.

**Arguments**
  * `attribute` `PropertyPath`
  * `value` `any|Array<any>` If it is an array, the values will be spread across the edges of the list.
Otherwise the value will be assigned to all edges.
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<EdgeList>`

## `edgeList.setAttributes(attributes[, options])`

Set the individual attributes of all the edges in the list.

**Arguments**
  * `attributes` `EdgeAttributesValue|Array<EdgeAttributesValue>` If a single attribute is specified, it is applied to all edges.
If an array is specified, each index of the array is assigned to the corresponding edge.
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<EdgeList>`

## `edgeList.setData([property], values)`

Set the specified data property of the edges. If no property is specified, update the whole data object.

**Arguments**
  * `property`(optional) `PropertyPath` Path of the data property to update.
  * `values` `Array<any>|function(edge: Edge): any` If it's an array, each value is assigned to the corresponding edge,
meaning the array must have the same length as the EdgeList. If it's a function, it will be applied to each edge to

**Returns**
  *  `EdgeList` Determine which value to assign.

## `edgeList.setDisabled(active)`

Add or remove the edges from the disabled state.

**Arguments**
  * `active` `boolean|Array<boolean>` Whether to disable or enable the edges.

## `edgeList.setSelected(active)`

**Arguments**
  * `active` `boolean|Array<boolean>` Whether to select or unselect the edges.

## `edgeList.setVisible()`

Call `setVisible` on each edge in the list.

## `edgeList.slice([start][, end])`

Returns a new EdgeList which contains only the edges from index `start` to `end` (excluding `end`).

**Arguments**
  * `start`(optional) `number`
  * `end`(optional) `number`

**Returns**
  *  `EdgeList`

## `edgeList.some(callback)`

**Arguments**
  * `callback` `function(edge: Edge, index: number, list: EdgeList): boolean`

**Returns**
  *  `boolean`

## `edgeList.subtract([list])`

**Arguments**
  * `list`(optional) `EdgeList` Returns a new EdgeList which does not contain any element from list

**Returns**
  *  `EdgeList`

## `edgeList.toArray()`

Returns an array of edges from the EdgeList.

**Returns**
  *  `Array<Edge>`

## `edgeList.toJSON([options])`

Runs `toJSON` on all the edges in the list and returns the list of objects.

**Arguments**
  * `options`(optional) `object`
      * `attributes`(optional) `Array<PropertyPath>|"all"`[=[]]
      * `data`(optional) `function (data: any): any`

**Returns**
  *  `Array<RawEdge>`

## `edgeList.toList()`

Returns itself.

**Returns**
  *  `EdgeList`

# Node

Ogma node entity class. Use this class to manipulate nodes, change data, attributes, etc.
 * ## `node.isNode`

Read-only property that is always `true`.

## `node.addClass(className[, options])`

Add the specified class to the node.

**Arguments**
  * `className` `string`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Node>`

## `node.addClasses(classNames[, options])`

Add the specified classes to the node.

**Arguments**
  * `classNames` `Array<string>`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Node>`

## `node.get(index)`

Convenience method to make Nodes and NodeLists more uniform

**Arguments**
  * `index` `number`

**Returns**
  *  `Node|undefined`

## `node.getAdjacentEdges([options])`

Returns the list of adjacent edges of the node.

**Arguments**
  * `options`(optional) `AdjacencyOptions`

**Returns**
  *  `EdgeList`

## `node.getAdjacentElements([options])`

Returns the list of adjacent nodes of the node and the edges connected to it.

**Arguments**
  * `options`(optional) `AdjacencyOptions`

**Returns**
  *  `SubGraph`

## `node.getAdjacentNodes([options])`

Returns the list of adjacent nodes of the node.

**Arguments**
  * `options`(optional) `AdjacencyOptions`

**Returns**
  *  `NodeList`

## `node.getAttribute(attributeName)`

Returns the value of the specified attribute for the node.

**Arguments**
  * `attributeName` `PropertyPath` Attribute to retrieve.

**Returns**
  *  `any`

## `node.getAttributes([attributeNames])`

Returns an object containing the specified attributes for the node.

**Arguments**
  * `attributeNames`(optional) `Array<PropertyPath>` List of attributes to include in the object. If not specified,  includes all the node attributes.

**Returns**
  *  `NodeAttributes`

## `node.getBoundingBox([options])`

Returns the bounding box of the node, in graph coordinates.

**Arguments**
  * `options`(optional) `object`
      * `includeTexts`(optional) `boolean`[default: false] Wether or not take texts in account in the bouding box

**Returns**
  *  `BoundingBox`

## `node.getClassList()`

Returns the list of classes that the node has.

**Returns**
  *  `Array<string>`

## `node.getConnectedComponent([options])`

**Arguments**
  * `options`(optional) `object`
      * `filter`(optional) `Filter`[default: &#039;visible&#039;]
      * `returnIds`(optional) `boolean`[default: false] Return node ids instead of Nodes

**Returns**
  *  `NodeList`

## `node.getData([property])`

Retrieve the specified data property of the node. If no property is
specified, retrieve the whole data object. This method method returns the
internal data object; modifying it could cause unexpected behavior.

**Arguments**
  * `property`(optional) `PropertyPath`

**Returns**
  *  `any`

## `node.getDegree([options])`

Retrieve the number of neighbors of the node.

**Arguments**
  * `options`(optional) `object|EdgeDirection`
      * `direction`(optional) `EdgeDirection`[default: "both"] Direction of the edges to follow.
      * `filter`(optional) `Filter`[default: "visible"] Indicates which edges to take into account

**Returns**
  *  `number`

## `node.getGeoCoordinates()`

Returns node's geographical coordinate

**Returns**
  *  `GeoCoordinate`

## `node.getId()`

Returns the id of the node.

**Returns**
  *  `NodeId`

## `node.getMetaNode()`

If the node is grouped inside a meta-node, returns this meta-node.
Otherwise, returns null.

**Returns**
  *  `Node|null`

## `node.getPosition()`

Retrieve the position of the node. This is strictly equivalent to `node.getAttributes(['x', 'y'])`.

**Returns**
  *  `{x: number, y: number}`

## `node.getSubNodes()`

If the node is a meta-node (result of a grouping), returns the list of
nodes that are part of the group
it represents. If it's not a meta-node, returns `null`.

**Returns**
  *  `NodeList|null`

## `node.getTransformation()`

Returns the transformation that created the node, if it is virtual.
Otherwise returns `null`.

**Returns**
  *  `Transformation|null`

## `node.hasClass(className)`

Indicates if the node has the specified class.

**Arguments**
  * `className` `string`

**Returns**
  *  `boolean`

## `node.isDisabled()`

Indicates if the node is currently disabled.

**Returns**
  *  `boolean`

## `node.isInView([options])`

Indicates if the node is visible in the current view.

**Arguments**
  * `options`(optional) `object`
      * `margin`(optional) `number`[default: 0] Tolerance in pixels.

**Returns**
  *  `boolean`

## `node.isSelected()`

Indicates if the node is currently selected.

**Returns**
  *  `boolean`

## `node.isVirtual()`

Indicates if the node was created by a transformation (`true`) or not
(`false`).

**Returns**
  *  `boolean`

## `node.isVisible()`

Indicates if the node is visible. A node is not visible if it has been filtered out, or if it is used in a
transformation.
/!\ A node with an opacity of 0 is considered visible!

**Returns**
  *  `boolean`

## `node.locate([options])`

Centers the view on the node.

**Arguments**
  * `options`(optional) `LocateOptions`

**Returns**
  *  `Promise<void>`

## `node.pulse([options])`

Highlights the node. It's a shorthand for the case when you
want the elements pulse for `number * (interval - 1) + duration` milliseconds.
It will also update the pulse attributes of the items with the one provided
in the `.pulse()` call.

**Arguments**
  * `options`(optional) `object`
      * `duration`(optional) `number`[default: 1000] Duration of a pulse (milliseconds)
      * `endColor`(optional) `Color|"inherit"`[default: "rgb(0,0,0,0.0)"] Ending color of the pulse
      * `endRatio`(optional) `number`[default: 2] Where the pulse ends, relative to the node siz (1 = at the node's border)
      * `interval`(optional) `number`[default: 800] Interval between two pulses (milliseconds)
      * `number`(optional) `number`[default: 1] Number of pulses
      * `startColor`(optional) `Color|"inherit"`[default: "rgb(0,0,0,0.6)"] Starting color of the pulse
      * `startRatio`(optional) `number`[default: 1] Where the pulse starts, relative to the node siz (1 = at the node's border)
      * `width`(optional) `number`[default: 50] Width of the pulse in pixels

**Returns**
  *  `Promise<Node>`

## `node.removeClass(className[, options])`

Remove the specified class from the node.

**Arguments**
  * `className` `string`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Node>`

## `node.removeClasses(classNames[, options])`

Remove the specified class from the node.

**Arguments**
  * `classNames` `Array<string>`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Node>`

## `node.resetAttributes([attributeNames][, options])`

Remove all attributes that have been applied through `setAttributes`. Original attributes or attributes applied by  the rules are not affected.

**Arguments**
  * `attributeNames`(optional) `Array<PropertyPath>` List of attributes to clear. If no attribute is specified, clear all  of them.
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Node>`

## `node.setAttribute(attribute, value[, options])`

Set the specified attribute of the node.

**Arguments**
  * `attribute` `PropertyPath`
  * `value` `any`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Node>`

## `node.setAttributes(attributes[, options])`

Set the individual attributes of the node.

**Arguments**
  * `attributes` `NodeAttributesValue` Attributes to update
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<Node>`

## `node.setData([property], value)`

Set the specified data property of the node. If no property is specified,
update the whole data object.

**Arguments**
  * `property`(optional) `PropertyPath`
  * `value` `any|function(node: Node): any`

**Returns**
  *  `Node`

## `node.setDisabled(active)`

Add or remove the node to/from the disabled state.

**Arguments**
  * `active` `boolean` Whether to disable or enable the node.

## `node.setGeoCoordinates(coords)`

Set geographical position of the node. Passing null will erase the
coordinates and remove the node from the visualisation in geo mode.

**Arguments**
  * `coords` `GeoCoordinate|null`

**Returns**
  *  `Promise<Node>`

## `node.setSelected(active)`

Add or remove the node to/from the selection.

**Arguments**
  * `active` `boolean` Whether to select or unselect the node.

## `node.setVisible(value)`

Hide or show the node.

**Arguments**
  * `value` `boolean` Whether to show or hide the node.

## `node.toJSON([options])`

Returns an object containing the id, attributes and data of the node.

**Arguments**
  * `options`(optional) `object`
      * `attributes`(optional) `Array<PropertyPath>|"all"`[=[]] List of
attributes to retrieve. By default, retrieve all attributes.
      * `data`(optional) `function (data: any): any` Function that takes the
node's data in input and return the data to retrieve. By default return
the whole object.

**Returns**
  *  `RawNode`

## `node.toList()`

Returns a new NodeList that contains only the node.

**Returns**
  *  `NodeList`

# NodeList

Ogma node list class. Use this class to manipulate a list of nodes, change data, attributes, etc.
 * ## `nodeList.isNode`

Read-only property that is always `true`.
 * ## `nodeList.size`

Read-only property that indicates the number of nodes in the list.

## `nodeList.addClass(className[, options])`

Add the specified class to the nodes.

**Arguments**
  * `className` `string`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<NodeList>`

## `nodeList.addClasses(classNames[, options])`

Add the specified classes to the nodes.

**Arguments**
  * `classNames` `Array<string>`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<NodeList>`

## `nodeList.concat(nodes)`

**Arguments**
  * `nodes` `NodeList`

**Returns**
  *  `NodeList`

## `nodeList.dedupe()`

Returns a new NodeList which does not contain any duplicate node.

**Returns**
  *  `NodeList`

## `nodeList.fillData([property], value)`

Set the specified data property of the nodes with the same value.

**Arguments**
  * `property`(optional) `PropertyPath|any` Path of the data property to update. If no property is specified, update the whole data object.
  * `value` `any` Value that will be assigned to all the nodes.

**Returns**
  *  `NodeList`

## `nodeList.filter(callback)`

**Arguments**
  * `callback` `function(node: Node, index: number): boolean`

**Returns**
  *  `NodeList`

## `nodeList.find(callback)`

**Arguments**
  * `callback` `function(node: Node, index: number, list: NodeList): boolean`

**Returns**
  *  `Node|undefined`

## `nodeList.forEach(callback)`

**Arguments**
  * `callback` `function(node: Node, index: number)`

## `nodeList.get(index)`

Returns the node at the specified index.

**Arguments**
  * `index` `number`

**Returns**
  *  `Node`

## `nodeList.getAdjacentEdges([options])`

Returns the list of adjacent edges of the nodes.

**Arguments**
  * `options`(optional) `AdjacencyOptions`

**Returns**
  *  `EdgeList`

## `nodeList.getAdjacentElements([options])`

Returns the list of adjacent nodes of the nodeList and the edges connected to it.

**Arguments**
  * `options`(optional) `AdjacencyOptions`

**Returns**
  *  `SubGraph`

## `nodeList.getAdjacentNodes([options])`

Returns the list of adjacent nodes of the nodes.

**Arguments**
  * `options`(optional) `AdjacencyOptions`

**Returns**
  *  `NodeList`

## `nodeList.getAttribute(attributeName)`

Returns an array containing the value of the specified attribute for each node.

**Arguments**
  * `attributeName` `PropertyPath` Attribute to retrieve.

**Returns**
  *  `Array<any>`

## `nodeList.getAttributes([attributes])`

Returns an array of objects containing the specified attributes for each node.

**Arguments**
  * `attributes`(optional) `Array<PropertyPath>` List of attributes to include in the object. If not specified, includes all the node attributes.

**Returns**
  *  `NodeAttributes[]`

## `nodeList.getBoundingBox([options])`

Returns the bounding box of the nodes, in graph coordinates.

**Arguments**
  * `options`(optional) `object`
      * `includeTexts`(optional) `boolean`[default: false] Wether or not take texts in account in the bouding box

**Returns**
  *  `BoundingBox`

## `nodeList.getClassList()`

Returns the list of classes that each node has.

**Returns**
  *  `Array<Array<string>>`

## `nodeList.getConnectedComponents([options])`

Returns weakly connected components of the list of nodes.

**Arguments**
  * `options`(optional) `object`
      * `filter`(optional) `Filter`[default: &#039;visible&#039;]
      * `returnIds`(optional) `boolean`[default: false] Return node ids instead of Nodes

**Returns**
  *  `Array<NodeList>`

## `nodeList.getData([property])`

Retrieve the specified data property. If no property is specified, retrieve the whole data object.
This method method returns the internal data object; modifying it could cause unexpected behavior.

**Arguments**
  * `property`(optional) `PropertyPath`

**Returns**
  *  `Array<any>`

## `nodeList.getDegree()`

Runs `getDegree` on each node in the list and returns the array of results.

## `nodeList.getGeoCoordinates()`

Returns geographical coordinates of all the nodes in the collection

**Returns**
  *  `Array<GeoCoordinate>`

## `nodeList.getId()`

Returns the id of each node.

**Returns**
  *  `Array<NodeId>`

## `nodeList.getMetaNode()`

Run `getMetaNode` on each node in the list and returns the array of results.

**Returns**
  *  `Array<Node|null>`

## `nodeList.getPosition()`

Retrieve the position of each node in the list. This is strictly equivalent to `nodeList.getAttributes(['x', 'y'])`.

**Returns**
  *  `Array<{x: number, y: number}>`

## `nodeList.getSubNodes()`

Run `getSubNodes` on all the nodes in the list and returns the array of results

**Returns**
  *  `Array<NodeList|null>`

## `nodeList.includes(node)`

Indicates if the `NodeList` contains the specified node.

**Arguments**
  * `node` `Node`

**Returns**
  *  `boolean`

## `nodeList.inverse()`

Returns a new NodeList containing all the visible nodes that are not in the list.

**Returns**
  *  `NodeList`

## `nodeList.isDisabled()`

Indicates for each node if it is disabled.

**Returns**
  *  `Array<boolean>`

## `nodeList.isSelected()`

Indicates for each node if it is selected.

**Returns**
  *  `Array<boolean>`

## `nodeList.isVisible()`

Call `isVisible` on each node in the list, and returns the array of results.

**Returns**
  *  `Array<boolean>`

## `nodeList.locate([options])`

Centers the view on the nodes.

**Arguments**
  * `options`(optional) `LocateOptions`

**Returns**
  *  `Promise<void>`

## `nodeList.map(callback)`

**Arguments**
  * `callback` `function(node: Node, index: number, list: NodeList): any`

**Returns**
  *  `Array<any>`

## `nodeList.pulse([options])`

Highlights the nodes with a pulse. It's a shorthand for the case when you
want the elements pulse for `number * (interval - 1) + duration` milliseconds.
It will also update the pulse attributes of the items with the one provided
in the `.pulse()` call.

**Arguments**
  * `options`(optional) `object`
      * `duration`(optional) `number`[default: 1000] Duration of a pulse (milliseconds)
      * `endColor`(optional) `Color|"inherit"`[default: "rgb(0,0,0,0.0)"] Ending color of the pulse
      * `endRatio`(optional) `number`[default: 2] Where the pulse ends, relative to the node siz (1 = at the node's border)
      * `interval`(optional) `number`[default: 800] Interval between two pulses (milliseconds)
      * `number`(optional) `number`[default: 1] Number of pulses
      * `startColor`(optional) `Color|"inherit"`[default: "rgb(0,0,0,0.6)"] Starting color of the pulse
      * `startRatio`(optional) `number`[default: 1] Where the pulse starts, relative to the node siz (1 = at the node's border)
      * `width`(optional) `number`[default: 50] Width of the pulse in pixels

**Returns**
  *  `Promise<NodeList>`

## `nodeList.reduce(callback, initialValue)`

**Arguments**
  * `callback` `function(accumulator: any, currentValue: Node, index: number): any`
  * `initialValue` `any`

**Returns**
  *  `any`

## `nodeList.removeClass(className[, options])`

Remove the specified class from the nodes.

**Arguments**
  * `className` `string`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<NodeList>`

## `nodeList.removeClasses(classNames[, options])`

Remove the specified class from the nodes.

**Arguments**
  * `classNames` `Array<string>`
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<NodeList>`

## `nodeList.resetAttributes([attributes][, options])`

Remove all attributes that have been applied through `setAttributes` of all the nodes in the list.
Original attributes or attributes applied by the rules are not affected.

**Arguments**
  * `attributes`(optional) `Array<PropertyPath>` List of attributes to clear. If no attribute is specified, clear all of them.
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<NodeList>`

## `nodeList.setAttribute(attribute, values[, options])`

Set the specified attribute of all the nodes in the list.

**Arguments**
  * `attribute` `PropertyPath`
  * `values` `any|Array<any>` If it is an array, the values will be spread across the nodes of the list.
Otherwise the value will be assigned to all nodes.
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<NodeList>`

## `nodeList.setAttributes(attributes[, options])`

Set the individual attributes of all the nodes in the list.

**Arguments**
  * `attributes` `NodeAttributesValue|Array<NodeAttributesValue>` If a single attribute is specified, it is applied to all nodes.
If an array is specified, each index of the array is assigned to the corresponding node.
  * `options`(optional) `AttributeAnimationOptions`

**Returns**
  *  `Promise<NodeList>`

## `nodeList.setData([property], values)`

Set the specified data property of the nodes. If no property is specified, update the whole data object.

**Arguments**
  * `property`(optional) `PropertyPath` Path of the data property to update.
  * `values` `Array<any>|function(node: Node): any` If it's an array, each value is assigned to the corresponding node,
meaning the array must have the same length as the NodeList. If it's a function, it will be applied to each node to
determine which value to assign.

**Returns**
  *  `NodeList`

## `nodeList.setDisabled(active)`

Add or remove the nodes from the disabled state.

**Arguments**
  * `active` `boolean|Array<boolean>` Whether to disable or enable the nodes.

## `nodeList.setGeoCoordinates(coordinates[, duration])`

Assign geographical coordinates to the nodes in collection

**Arguments**
  * `coordinates` `Array<GeoCoordinate|null>|null`
  * `duration`(optional) `number` Animation duration

**Returns**
  *  `Promise<NodeList>`

## `nodeList.setSelected(active)`

Add or remove the nodes from the selection.

**Arguments**
  * `active` `boolean|Array<boolean>` Whether to select or unselect the nodes.

## `nodeList.setVisible()`

Call `setVisible` on each node in the list.

## `nodeList.slice([start][, end])`

Returns a new NodeList which contains only the nodes from index `start` to `end` (excluding `end`).

**Arguments**
  * `start`(optional) `number`
  * `end`(optional) `number`

**Returns**
  *  `NodeList`

## `nodeList.some(callback)`

**Arguments**
  * `callback` `function(node: Node, index: number, list: NodeList): boolean`

**Returns**
  *  `boolean`

## `nodeList.sort(fn)`

**Arguments**
  * `fn` `(a: Node, b: Node): number` Sort function

## `nodeList.subtract([list])`

**Arguments**
  * `list`(optional) `NodeList` Returns a new NodeList which does not contain any element from list

**Returns**
  *  `NodeList`

## `nodeList.toArray()`

Returns an array of nodes from the NodeList.

**Returns**
  *  `Array<Node>`

## `nodeList.toJSON([options])`

Runs `toJSON` on all the nodes in the list and returns the list of objects.

**Arguments**
  * `options`(optional) `object`
      * `attributes`(optional) `Array<PropertyPath>|"all"`[=[]]
      * `data`(optional) `function (data: any): any`

**Returns**
  *  `Array<RawNode>`

## `nodeList.toList()`

Returns itself.

**Returns**
  *  `NodeList`

# geometry

Ogma static toolbox for geometry purposes

## `geometry.computeCentroid(points)`

Returns the average of the specified points.

**Arguments**
  * `points` `Point[]`

**Returns**
  *  `Point`

## `geometry.distance(x1, y1, x2, y2)`

Compute the distance between two points (x1, y1) and (x2, y2).

**Arguments**
  * `x1` `number`
  * `y1` `number`
  * `x2` `number`
  * `y2` `number`

**Returns**
  *  `number` Distance between the two points.

## `geometry.getNormalOnEdge(edge, t)`

Returns the normal vector of the edge at t.
Returns the normal at the source of the edge for t = 0 and at the target for t = 1

**Arguments**
  * `edge` `Edge` An edge
  * `t` `number` The interpolation value bounded in [0;1].

**Returns**
  *  `object` {x,y} The normal to the edge at t

## `geometry.getPointOnEdge(edge, t)`

Returns the point of the edge at t.
Returns the source of the edge for t = 0 and at the target for t = 1

**Arguments**
  * `edge` `Edge` An edge
  * `t` `number` The interpolation value bounded in [0;1].

**Returns**
  *  `object` {x,y} The position on the edge at t

## `geometry.getQuadraticCurveControlPoint(x1, y1, x2, y2, curvature, dest?)`

Returns the control point used to draw a curved edge.

**Arguments**
  * `x1` `number` X position of the source
  * `y1` `number` Y position of the source
  * `x2` `number` X position of the target
  * `y2` `number` Y position of the target
  * `curvature` `number` The curvature factor of the edge
  * `dest?` `Point` A Point to write in the result

**Returns**
  *  `Point` The control point used to draw a curved edge.

## `geometry.getTangentOnEdge(edge, t)`

Returns the tangent vector of the edge at t.
Returns the tangent at the source of the edge for t = 0 and at the target for t = 1

**Arguments**
  * `edge` `Edge` An edge
  * `t` `number` The interpolation value bounded in [0;1].

**Returns**
  *  `object` {x,y} The tangent to the edge at t

# geometry.lines

## `geometry.lines.getClosestPointOnSegment(px, py, ax, ay, bx, by)`

Get the closest point on a segment from a point

**Arguments**
  * `px` `number` Point x
  * `py` `number` Point y
  * `ax` `number` Segment start x
  * `ay` `number` Segment start y
  * `bx` `number` Segment end x
  * `by` `number` Segment end y

**Returns**
  *  `Point`

## `geometry.lines.lineAngle()`

Compute the angle formed by the line joining two points (x1, y1) and (x2, y2)

**Returns**
  *  `number`

## `geometry.lines.lineIntersection(ax0, ay0, ax1, ay1, bx0, by0, bx1, by1)`

Intersection between two lines (unbounded)

**Arguments**
  * `ax0` `number` The X coordinate of the start point of the first line.
  * `ay0` `number` The Y coordinate of the start point of the first line.
  * `ax1` `number` The X coordinate of the end point of the first line.
  * `ay1` `number` The Y coordinate of the end point of the first line.v
  * `bx0` `number` The X coordinate of the start point of the second line.
  * `by0` `number` The Y coordinate of the start point of the second line.
  * `bx1` `number` The X coordinate of the end point of the second line.
  * `by1` `number` The Y coordinate of the end point of the second line.

## `geometry.lines.segmentIntersection(ax0, ay0, ax1, ay1, bx0, by0, bx1, by1)`

Intersection between two segments

**Arguments**
  * `ax0` `number` The X coordinate of the start point of the first line.
  * `ay0` `number` The Y coordinate of the start point of the first line.
  * `ax1` `number` The X coordinate of the end point of the first line.
  * `ay1` `number` The Y coordinate of the end point of the first line.v
  * `bx0` `number` The X coordinate of the start point of the second line.
  * `by0` `number` The Y coordinate of the start point of the second line.
  * `bx1` `number` The X coordinate of the end point of the second line.
  * `by1` `number` The Y coordinate of the end point of the second line.

## `geometry.lines.segmentRectangleIntersection(ax, ay, bx, by, minX, minY, maxX, maxY)`

**Arguments**
  * `ax` `Number`
  * `ay` `Number`
  * `bx` `Number`
  * `by` `Number`
  * `minX` `Number`
  * `minY` `Number`
  * `maxX` `Number`
  * `maxY` `Number`

**Returns**
  *  `Array<Point>`

## `geometry.lines.twoSegmentsIntersect(p1, p2, p3, p4[, excludeBoundaries])`

Indicates if two segments intersect.

**Arguments**
  * `p1` `Point` First extremity of the first segment
  * `p2` `Point` Second extremity of the first segment
  * `p3` `Point` First extremity of the second segment
  * `p4` `Point` Second extremity of the second segment
  * `excludeBoundaries`(optional) `boolean`[default: false] If true, the segment will not be considered as
                                           intersecting if the intersection point is one
                                           of their extremities

**Returns**
  *  `boolean`

# parse

Ogma static toolbox for data imports, use as `Ogma.parse.*`.

## `parse.gexf(content)`

Parse a GEXF string and return the raw graph.

**Arguments**
  * `content` `string`

**Returns**
  *  `Promise<RawGraph>`

## `parse.gexfFromUrl(url)`

Fetch and parse a GEXF file and return the raw graph.

**Arguments**
  * `url` `string`

**Returns**
  *  `Promise<RawGraph>`

## `parse.graphml(content[, xmlparser])`

Parse a GraphML string and return the raw graph.

**Arguments**
  * `content` `string` GraphML string
to transform custom JSON format into Ogma's `RawGraph`
  * `xmlparser`(optional) `DOMParser` DOMParser instance to use for parsing the XML

**Returns**
  *  `Promise<RawGraph>`

## `parse.graphmlFromUrl(url[, xmlparser])`

Fetch and parse a GraphML file and return the raw graph.

**Arguments**
  * `url` `string`
  * `xmlparser`(optional) `DOMParser` DOMParser instance to use for parsing the XML

**Returns**
  *  `Promise<RawGraph>`

## `parse.janus(content)`

Parse the result of a JanusGraph query into an Ogma graph.

**Arguments**
  * `content` `object` Response of the gremlin-client library ("gremlin-client")

**Returns**
  *  `Promise<RawGraph>`

## `parse.json(content[, transform])`

Parse a JSON string and return the raw graph.

**Arguments**
  * `content` `string`
  * `transform`(optional) `function(json: object | unknown[]): RawGraph` Function
to transform custom JSON format into Ogma's `RawGraph`

**Returns**
  *  `Promise<RawGraph>`

## `parse.jsonFromUrl(url[, transform])`

Fetch and parse a JSON file and return the raw graph.

**Arguments**
  * `url` `string`
  * `transform`(optional) `function(json: object | unknown[]): RawGraph` Function
to transform custom JSON format into Ogma's `RawGraph`

**Returns**
  *  `Promise<RawGraph>`

## `parse.mtx(content)`

Parse a Matrix Market file (.mtx)

**Arguments**
  * `content` `string`

**Returns**
  *  `Promise<RawGraph>`

## `parse.mtxFromUrl(url)`

Loads and parses a Matrix Market file (.mtx)

**Arguments**
  * `url` `string`

**Returns**
  *  `Promise<RawGraph>`

## `parse.neo4j(content)`

**Arguments**
  * `content` `object` Response of the Neo4j Bolt driver ("neo4j-javascript-driver")

**Returns**
  *  `Promise<RawGraph>`

# StyleClass

Represents a visual class that can be applied to nodes and edges.

## `styleClass.add(elements)`

Add the class to the specified node(s)/edge(s). Equivalent to to `elements.addClass(myClass.getName())`.

**Arguments**
  * `elements` `Node|Edge|NodeList|EdgeList`

## `styleClass.clearEdges([filter])`

Remove the class from all edges.

**Arguments**
  * `filter`(optional) `Filter` Filter to apply to edges

## `styleClass.clearNodes([filter])`

Remove the class from all nodes.

**Arguments**
  * `filter`(optional) `Filter` Filter to apply to nodes

## `styleClass.destroy()`

Remove the class from all nodes and edges and delete it.

**Returns**
  *  `Promise<void>`

## `styleClass.getDefinition()`

Returns the node and edge attributes associated with the class.

**Returns**
  *  `StyleClassDefinition`

## `styleClass.getEdges([filter])`

Returns the list of edges that have the class. Does not include filtered edges.

**Arguments**
  * `filter`(optional) `Filter` Filter to apply to edges

**Returns**
  *  `EdgeList`

## `styleClass.getIndex()`

Returns the index of the class in the internal class array. A higher index class is applied after a lower
index class.

**Returns**
  *  `number`

## `styleClass.getName()`

Returns the name of the class.

**Returns**
  *  `string`

## `styleClass.getNodes(Filter)`

Returns the list of nodes that have the class. Does not include filtered nodes.

**Arguments**
  * `Filter` [filter] filter to apply to nodes

**Returns**
  *  `NodeList`

## `styleClass.remove(elements)`

Remove the class from the specified node(s)/edge(s). Equivalent to to `elements.removeClass(myClass.getName())`.

**Arguments**
  * `elements` `Node|Edge|NodeList|EdgeList`

## `styleClass.setIndex(index)`

Set the index of the class in the internal class array. A higher index class is applied after a lower
index class.

**Arguments**
  * `index` `number`

## `styleClass.update(options)`

Updates the attributes assigned to nodes and edges that have that class.

**Arguments**
  * `options` `object`
      * `edgeAttributes`(optional) `EdgeAttributesValue`
      * `edgeDependencies`(optional) `EdgeDependencies`
      * `edgeOutput`(optional) `EdgeOutput`
      * `fullOverwrite`(optional) `boolean`[default: false] Indicates if the specified attributes should be merge with current
ones (false) or if the specified attributes should entirely replace the current ones.
      * `nodeAttributes`(optional) `NodeAttributesValue`
      * `nodeDependencies`(optional) `NodeDependencies`
      * `nodeOutput`(optional) `NodeOutput`

# StyleRule

Controller class for a style rule, used to update and manage the attributes and selectors associated with the rule.

## `styleRule.destroy()`

Delete the rule. After this is called, a call to any method on this object
will throw an error.

**Returns**
  *  `Promise<void>`

## `styleRule.getId()`

Returns the unique positive integer rule id associated with the rule.

**Returns**
  *  `number`

## `styleRule.getIndex()`

Retrieve the position of the rule in the internal rule list. Rules with a higher index are applied after rules
with a lower index.

**Returns**
  *  `number`

## `styleRule.refresh()`

Refresh the rule for all nodes.

**Returns**
  *  `Promise<void>`

## `styleRule.setIndex(index)`

Assign the position of the rule in the internal rule list. Rules with a higher index are applied after rules
with a lower index.

**Arguments**
  * `index` `number`

**Returns**
  *  `Promise<void>`

## `styleRule.update(options)`

Updates the attributes and selectors associated with the rule.

**Arguments**
  * `options`
      * `edgeAttributes`(optional) `EdgeAttributesValue`
      * `edgeDependencies`(optional) `EdgeDependencies`
      * `edgeOutput`(optional) `EdgeOutput`
      * `edgeSelector`(optional) `EdgeSelector|null`
      * `fullOverwrite`(optional) `boolean`[default: false] Indicates if
the specified attributes should be merged with current ones (false) or if
the specified attributes should entirely replace the current ones.
      * `nodeAttributes`(optional) `NodeAttributesValue`
      * `nodeDependencies`(optional) `NodeDependencies`
      * `nodeOutput`(optional) `NodeOutput`
      * `nodeSelector`(optional) `NodeSelector|null`

**Returns**
  *  `Promise<void>`

## `styleRule.whenApplied()`

Call the specified function when the rule is applied for the first time.

**Returns**
  *  `Promise<StyleRule>`

# NonObjectPropertyWatcher

Retrieved from `ogma.schema.watchNodeNonObjectProperty()` or `ogma.schema.watchEdgeNonObjectProperty()`.

## `nonObjectPropertyWatcher.destroy()`

Stops the watcher from being updated

## `nonObjectPropertyWatcher.getPath()`

Returns the path of the data property being watched

**Returns**
  *  `Array<string>`

## `nonObjectPropertyWatcher.getPropertyInfo()`

Retrieve some information on the property being watched.

**Returns**
  *  `PropertyInformation`

## `nonObjectPropertyWatcher.onUpdate(handler)`

Triggers the specified function when the property is updated,

**Arguments**
  * `handler` `function (info: PropertyInformation)`

# ObjectPropertyWatcher

Retrieved from `ogma.schema.watchNodeProperties()` or `ogma.schema.watchEdgeProperties()`.

## `objectPropertyWatcher.destroy()`

Stops the watcher from being updated.

## `objectPropertyWatcher.getPath()`

Returns the path of the data property being watched

**Returns**
  *  `Array<string>`

## `objectPropertyWatcher.getProperties()`

Returns the names of the sub-property of the watched property.

**Returns**
  *  `Array<string>`

## `objectPropertyWatcher.getPropertyInfo(name)`

Retrieve some information on the specified sub-property.

**Arguments**
  * `name` `string`

**Returns**
  *  `PropertyInformation|null`

## `objectPropertyWatcher.onPropertyAdded(handler)`

Triggers the specified function when a new sub-property is added (at least one node has a value for it).

**Arguments**
  * `handler` `function (property: string, info: PropertyInformation)`

**Returns**
  *  `ObjectPropertyWatcher`

## `objectPropertyWatcher.onPropertyRemoved(handler)`

Triggers the specified function when a new sub-property is removed (no node has a value for it).

**Arguments**
  * `handler` `function (property: string, info: PropertyInformation)`

**Returns**
  *  `ObjectPropertyWatcher`

## `objectPropertyWatcher.onPropertyUpdated(handler)`

Triggers when a sub-property of the watched property is updated.

**Arguments**
  * `handler` `function (property: string, info: PropertyInformation)`

**Returns**
  *  `ObjectPropertyWatcher`

# PropertyInformation

Retrieved from `watcher.getPropertyInfo()`.

## `propertyInformation.getBoundaries()`

If there is at least one numerical value for this property, returns the
minimum and maximum value for this property across the graph. Returns
`null` if there is no numerical value for that property.

**Returns**
  *  `null|{min: number, max: number}`

## `propertyInformation.getCount()`

Returns the number of nodes/edges for which the property is different from `undefined`

**Returns**
  *  `number`

## `propertyInformation.getType()`

Returns the type of the property.

**Returns**
  *  `"any"|"number"|"undefined"` If there is at least one non-number (excluding `undefined`) value, returns `'any'`.
If there are only numerical values for that property, returns `'number'`. If no node/edge has a value for that property,
returns `'undefined'`.

## `propertyInformation.getValueCount(value)`

Returns the number of nodes/edges for which the property has the specified value.

**Arguments**
  * `value` `any`

**Returns**
  *  `number`

## `propertyInformation.getValues()`

Returns the different values for this property.

**Returns**
  *  `Array<any>`

## `propertyInformation.isNode()`

Indicates if it is a node or edge property.

**Returns**
  *  `boolean`

# Transformation

Controller for an individual transformation on the graph, such as grouping, filtering etc.

## `transformation.destroy(duration)`

Remove the transformation over the specified amount of time. After this methods is called, the transformation is not manipulable anymore (cannot be enabled again for example). Note: the animation will not be played if the transformation is not the last in the transformation stack, because the transformations are sequential

**Arguments**
  * `duration` `number`

**Returns**
  *  `Promise<void>`

## `transformation.disable([duration])`

Disable the transformation over the specified amount of time.

**Arguments**
  * `duration`(optional) `number`[default: 0] Animation duration in ms

**Returns**
  *  `Promise<void>`

## `transformation.enable([duration])`

Enable the transformation over the specified amount of time.

**Arguments**
  * `duration`(optional) `number`[default: 0] Animation duration in ms

**Returns**
  *  `Promise<void>`

## `transformation.getId()`

Returns the id of the transformation, a unique strictly positive integer.

**Returns**
  *  `number`

## `transformation.getIndex()`

Retrieves the index of the transformation in the pipeline.

**Returns**
  *  `number`

## `transformation.getName()`

Returns the name of the transformation.

**Returns**
  *  `"node-filter"|"edge-filter"|"node-grouping"|"edge-grouping"`

## `transformation.isEnabled()`

Returns true if the transformation is currently enabled

**Returns**
  *  `boolean`

## `transformation.refresh()`

Refresh the transformation.

**Returns**
  *  `Promise<void>`

## `transformation.setIndex(index)`

Set the index of the transformation in the pipeline. The transformation with the lower index is applied first,
the one with the higher index is applied last.

**Arguments**
  * `index`

**Returns**
  *  `Promise<void>`

## `transformation.setOptions(index)`

Set the index of the transformation in the pipeline. The transformation with the lower index is applied first,
the one with the higher index is applied last.

**Arguments**
  * `index`

**Returns**
  *  `Promise<void>`

## `transformation.toggle([duration])`

Toggle the transformation over the specified amount of time.

**Arguments**
  * `duration`(optional) `number`[default: 0] Animation duration in ms

**Returns**
  *  `Promise<void>`

## `transformation.whenApplied()`

Returns a Promise that resolves the first time the transformation is applied.

**Returns**
  *  `Promise<void>`

## `NodeAttributes`

**Type:**  `object`

**Properties**
  * `badges`(optional) `object`
      * `bottomLeft`(optional) `Badge`
      * `bottomRight`(optional) `Badge`
      * `topLeft`(optional) `Badge`
      * `topRight`(optional) `Badge`

  * `color`(optional) `Color|Array<Color>`[default: "grey"] Color of the node
  * `detectable`(optional) `boolean`[default: true] Indicates if the node is detectable by the mouse.
  * `draggable`(optional) `boolean`[default: true] Indicates if the node is draggable by the user
  * `halo`(optional) `object|Color|null` If not an object, alias for `halo.color`
      * `color`(optional) `Color`[default: null] Color of the halo
      * `hideNonAdjacentEdges`(optional) `boolean`[default: false] If true, the halo hides edges which don't have at least one extremity with `halo.hideNonAdjacentEdges` to `true`.
      * `scalingMethod`(optional) `ScalingMethod`[default: "fixed"] Indicates if the halo width should be multiplied by the zoom when the node is displayed.
      * `strokeColor`(optional) `Color`[default: null] Color of the stroke of the halo
      * `strokeWidth`(optional) `PixelSize`[default: 1] Width of the stroke of the halo
      * `width`(optional) `PixelSize`[default: 50] Width of the halo, in pixels

  * `icon`(optional) `object|TextContent` If not an object, alias for `icon.content`
      * `color`(optional) `Color`[default: "black"] Color used to display the icon text
      * `content`(optional) `TextContent` Text to display inside the icon
      * `font`(optional) `string`[default: "Arial"] Font used to display the icon text
      * `minVisibleSize`(optional) `number`[default: 12] If the node diameter on screen is less than this value, the icon will not be shown
      * `scale`(optional) `number`[default: 0.7] Text size relative to the node diameter
      * `style`(optional) `FontStyle`[default: "normal"] Style applied to the icon.

  * `image`(optional) `object|string|null` If not an object, alias for `image.url`
      * `fit`(optional) `boolean`[default: true] Indicates if the image should be rescaled to fit the node
      * `minVisibleSize`(optional) `number`[default: 12] If the node diameter on screen is less than this value, the image will not be shown
      * `scale`(optional) `number`[default: 1] Size of the image relative to the node diameter
      * `tile`(optional) `boolean`[default: false] If the image is smaller than the node, indicates if the image should be duplicated to fill the node. If true, `fit` will be considered to be `false`.
      * `url`(optional) `string|null`[default: null] URL of the image to display

  * `innerStroke`(optional) `object|Color|"inherit"` If not an object, alias for `innerStroke.color`
      * `color`(optional) `Color|"inherit"`[default: "white"] Color of the node's inner stroke
      * `minVisibleSize`(optional) `number`[default: 12] If the node diameter on screen is less than this value, the inner stroke will not be shown
      * `scalingMethod`(optional) `ScalingMethod`[default: "fixed"] Indicates if the inner stroke width should be multiplied by the zoom when the node is displayed.
      * `width`(optional) `PixelSize`[default: 2] Width of the node's inner stroke

  * `layer`(optional) `LayerValue`[default: 0] Z-index of the node. Integer value between 1 and 3.
  * `layoutable`(optional) `boolean`[default: true] Indicates if the node movable by the layout algorithm
                                                                            Some of the layouts (ForceLink) would take the non-movable nodes
                                                                            into account by making other nodes avoid them, others
                                                                            would just ignore them in calculations.
  * `opacity`(optional) `OpacityValue`[default: 1] Opacity of the node.
  * `outerStroke`(optional) `object|Color|"inherit"` If not an object, alias for `outerStroke.color`
      * `color`(optional) `Color|"inherit"`[default: null] Color of the node's outer stroke
      * `minVisibleSize`(optional) `number`[default: 0] If the node diameter on screen is less than this value, the outer stroke will not be shown
      * `scalingMethod`(optional) `ScalingMethod`[default: "fixed"] Indicates if the outer stroke width should be multiplied by the zoom when the node is displayed.
      * `width`(optional) `PixelSize`[default: 5] Width of the node's outer stroke

  * `outline`(optional) `object|boolean` If not an object, alias for `outline.enabled`
      * `color`(optional) `Color`[default: "rgba(0, 0, 0, 0.36)"] Color of the outline
      * `enabled`(optional) `boolean`[default: false] Indicates if the outline should be visible
      * `minVisibleSize`(optional) `number`[default: 12] If the node diameter on screen is less than this value, the outline will ne be shown

  * `pulse`(optional) `object|null`
      * `duration`(optional) `number`[default: 1000] Lifespan of one pulse ripple (milliseconds).
      * `enabled`(optional) `boolean`[default: false] If true, shows animated pulse around the node.
      * `endColor`(optional) `Color`[default: "rgba(0, 0, 0, 0)"] End color of the pulse
      * `endRatio`(optional) `number`[default: 2] Where the pulse ends, relative to the node size (2 = 2x size of the node)
      * `interval`(optional) `number`[default: 800] Interval between two pulses (milliseconds).
      * `startColor`(optional) `Color`[default: "rgba(0, 0, 0, 0.6)"] Starting color of the pulse
      * `startRatio`(optional) `number`[default: 1] Where the pulse starts, relative to the node size (1 = at the node's border)
      * `width`(optional) `number`[default: 50] Width of the pulse in pixels

  * `radius`(optional) `PixelSize`[default: 5] Indicates the radius of the node (graph size)
  * `scalingMethod`(optional) `ScalingMethod`[default: "scaled"] Indicates if the radius should be multiplied by the zoom when the node is displayed.
  * `shape`(optional) `NodeShape`[default: "circle"] Shape of the node
  * `text`(optional) `object|TextContent` If not an object, alias for `text.content`
      * `align`(optional) `TextAlign`[default: "center"] Alignment of the text (for multi-line texts)
      * `backgroundColor`(optional) `Color|"inherit"|null`[default: null] Background color of the text
      * `color`(optional) `Color|"inherit"`[default: "black"] Color of the text
      * `content`(optional) `TextContent`[default: null] Text to display
      * `font`(optional) `string`[default: "Arial"] Font used to display the text
      * `margin`(optional) `PixelSize`[default: 10] Additional space (in pixels) between the node and the text
      * `maxLineLength`(optional) `number`[default: 0] If > 1, lines that have more characters than this value will be split across multiple lines. Affects both primary and secondary texts.
      * `minVisibleSize`(optional) `number`[default: 24] If the node diameter on screen is less than this value, the text will not be shown
      * `padding`(optional) `PixelSize`[default: 2] Space between the text and its background's edge, in pixels
      * `position`(optional) `"right"|"left"|"top"|"bottom"|"center"`[default: "bottom"] Position of the text relative to the node
      * `scale`(optional) `number`[default: 0.1] Text size relative to the node diameter
      * `scaling`(optional) `boolean`[default: false] Indicates if the `size` property (false) or the `scale` property (true) must be used to compute the text size. Affects both primary and secondary text.
      * `secondary`(optional) `object|TextContent` If not an object, alias for `text.secondary.content`
          * `align`(optional) `TextAlign`[default: "center"] Alignment of the secondary text (for multi-line texts)
          * `backgroundColor`(optional) `Color|"inherit"|null`[default: null] Background color of the secondary text
          * `color`(optional) `Color|"inherit"`[default: "black"] Color of the secondary text
          * `content`(optional) `TextContent`[default: null] Secondary text content. The secondary text is always displayed below the node.
          * `font`(optional) `string`[default: "Arial"] Font used to display the secondary text
          * `margin`(optional) `PixelSize`[default: 2] Space (in pixels) on top of the secondary text.
          * `minVisibleSize`(optional) `number`[default: 24] If the node diameter on screen is less than this value, the secondary text will not be shown
          * `padding`(optional) `PixelSize`[default: 2] Space (in pixels) between the text and its background's edge
          * `scale`(optional) `number`[default: 0.08] Secondary text size (relative to the node diameter)
          * `size`(optional) `PixelSize`[default: 10] Secondary text size (in pixels)
          * `style`(optional) `FontStyle`[default: "normal"] Secondary text style

      * `size`(optional) `PixelSize`[default: 12] Text size (in pixels)
      * `style`(optional) `FontStyle`[default: "normal"] Style applied to the text
      * `tip`(optional) `boolean`[default: true] Indicates if the margin between the text and the background should be filled with a small arrow pointing towards the node

  * `x`(optional) `number`[default: 0] X coordinate of the node (graph space)
  * `y`(optional) `number`[default: 0] Y coordinate of the node (graph space)

## `EdgeAttributes`

**Type:**  `object`

Default values indicate the system values (when an edge has not been assigned any value for that attribute).

**Properties**
  * `adjustAnchors`(optional) `boolean`[default: true] If true, the edge's extremities' badges and shape will be taken into account when displaying it.
                                                                       Edges that ends with an arrow will stop at the node's badge/corner.
  * `color`(optional) `Color|"source"|"target"`[default: "grey"] Color of the edge
  * `detectable`(optional) `boolean`[default: true] Indicates if the edge is detectable by the mouse.
  * `halo`(optional) `object|Color` If not an object, alias to `halo.color`
      * `color`(optional) `Color`[default: null] Color of the halo
      * `scalingMethod`(optional) `ScalingMethod`[default: "fixed"] Indicates if the halo width should be multiplied by the zoom when the edge is displayed.
      * `width`(optional) `PixelSize`[default: 10] Width of the halo, in pixels

  * `layer`(optional) `LayerValue`[default: 0] Z-index of the node. Integer value between -1 and 3.
  * `minVisibleSize`(optional) `number`[default: 0] If the edge width on screen is less than this value, it will not be displayed
  * `opacity`(optional) `OpacityValue`[default: 1] Opacity of the edge. 0 = transparent, 1 = opaque.
  * `outline`(optional) `object|boolean` If not an object, alias to `outline.enabled`
      * `color`(optional) `Color`[default: "rgba(0, 0, 0, 0.36)"] Color of the outline
      * `enabled`(optional) `boolean`[default: false] Indicates if the outline should be visible
      * `minVisibleSize`(optional) `number`[default: 0] If the edge width on screen is less than this value, the outline will ne be shown

  * `pulse`(optional) `object`
      * `duration`(optional) `number`[default: 1000] Lifespan of one pulse ripple (milliseconds).
      * `enabled`(optional) `boolean`[default: false] If true, shows animated pulse around the edge.
      * `endColor`(optional) `Color`[default: "rgba(0, 0, 0, 0)"] End color of the pulse
      * `endRatio`(optional) `number`[default: 2] Where the pulse ends, relative to the edge width (2 = 2x width of the edge)
      * `interval`(optional) `number`[default: 800] Interval between two pulses (milliseconds).
      * `startColor`(optional) `Color`[default: "rgba(0, 0, 0, 0.6)"] Starting color of the pulse
      * `startRatio`(optional) `number`[default: 1] Where the pulse starts, relative to the edge width (1 = at the edge border)
      * `width`(optional) `number`[default: 50] Width of the pulse in pixels

  * `scalingMethod`(optional) `ScalingMethod`[default: "scaled"] Indicates if the edge width should be multiplied by the zoom when the edge is displayed.
  * `shape`(optional) `object|PredefinedEdgeShape`
      * `body`(optional) `EdgeType`[default: "line"] Shape of the edge
      * `head`(optional) `EdgeExtremity`[default: null] Head of the edge
      * `style`(optional) `EdgeStyle`[default: "plain"] Style of the edge
      * `tail`(optional) `EdgeExtremity`[default: null] Tail of the edge

  * `stroke`(optional) `object`
      * `color`(optional) `Color|"inherit"`[default: "inherit"] Color of the edge stroke. If it is "inherit", uses the same color as for the body.
      * `minVisibleSize`(optional) `number`[default: 0] If the edge width on screen is less than this value, the stroke will not be displayed
      * `width`(optional) `PixelSize`[default: 0] Stroke width, in pixels.

  * `strokeWidth`(optional) `PixelSize` Alias for `stroke.width`
  * `text`(optional) `object|TextContent` If not an object, alias to `text.content`
      * `adjustAngle`(optional) `boolean`[default: true] In case the edge is shorter than the text, indicates if the text should be displayed horizontally. Only works for non-scaled texts.
      * `align`(optional) `TextAlign`[default: "center"] Alignment of the text (for multi-line texts)
      * `backgroundColor`(optional) `Color|"inherit"`[default: null] Background color of the text
      * `color`(optional) `Color`[default: "black"] Color of the text
      * `content`(optional) `TextContent`[default: null] Text to display
      * `font`(optional) `string`[default: "Arial"] Font used to display the text
      * `margin`(optional) `PixelSize`[default: 2] Space between the text and the edge, in pixels. Ignored if `text.position` is "centered".
      * `maxLineLength`(optional) `number`[default: 0] If > 1, lines that have more characters than this value will be split across multiple lines. Affects both primary and secondary texts.
      * `minVisibleSize`(optional) `PixelSize`[default: 4] If the edge width on screen is less than this value, the text will not be shown
      * `padding`(optional) `PixelSize`[default: 2] Space between the text and its background's edge, in pixels
      * `position`(optional) `EdgeTextPosition`[default: "shifted"] Text position relative to the edge: centered or shifted. Centered places the text on top of the edge and ignores margin
      * `scale`(optional) `number`[default: 1] Text size relative to the edge width
      * `scaling`(optional) `boolean`[default: false] Indicates if the `size` property (false) or the `scale` property (true) must be used to compute the text size
      * `secondary`(optional) `object|TextContent` If not an object, alias to `text.secondary.content`
          * `align`(optional) `TextAlign`[default: "center"] Alignment of the secondary text (for multi-line texts)
          * `backgroundColor`(optional) `Color|"inherit"`[default: null] Background color of the secondary text
          * `color`(optional) `Color`[default: "black"] Color of the secondary text
          * `content`(optional) `TextContent`[default: null] Text to display under the primary text
          * `font`(optional) `string`[default: "Arial"] Font used to display the secondary text
          * `margin`(optional) `PixelSize`[default: 2] Space between the secondary text and the edge
          * `minVisibleSize`(optional) `PixelSize`[default: 4] If the edge width on screen is less than this value, the secondary text will not be shown
          * `padding`(optional) `PixelSize`[default: 2] Space between the secondary text and its background's edge, in pixels
          * `scale`(optional) `number`[default: 0.8] Secondary text size (relative to the edge width)
          * `size`(optional) `PixelSize`[default: 12] Secondary text size (in pixels)
          * `style`(optional) `FontStyle`[default: "normal"] Secondary text style

      * `size`(optional) `PixelSize`[default: 12] Text size (in pixels)
      * `style`(optional) `FontStyle`[default: "normal"] Style applied to the text

  * `width`(optional) `PixelSize`[default: 1] Width of the edge (graph space)

## `AdjacencyOptions`

**Type:**  `object`

**Properties**
  * `options`(optional)
      * `bothExtremities`(optional) `boolean`[default: false] Relevant only for `getAdjacentEdges`. If `true`, only edges for
which both extremities are in the `NodeList` are retrieved.
      * `direction`(optional) `"both"|"in"|"out"`[default: "both"] Direction of the edges to follow.
      * `filter`(optional) `Filter`[default: "visible"] Indicates what kind of elements should be retrieved.
      * `policy`(optional) `"union"|"include-sources"|"exclude-sources"`[default: "union"] If "include-sources", the source node(s) will be added to the result.
If "exclude-sources", the source node(s) will be removed from the result.
If "union", the result is not modified.
This parameter is ignored when retrieving adjacent edges.

## `AttributeAnimationOptions`

**Type:**  `object|number`

If a number is specified,
it specifies the `duration` property.

**Properties**
  * `duration`(optional) `number`[default: 0] Indicates the duration of the attribute
transition, in milliseconds.
  * `easing`(optional) `Easing`[default: "linear"] Indicates the animation easing

## `Badge`

**Type:**  `object`

Node badge attributes

**Properties**
  * `color`(optional) `Color`[default: "white"] Fill color of the badge.
  * `image`(optional) `object|null|string` If not an object, alias to `image.url`
      * `scale`(optional) `number`[default: 1] Size of the image relative to the badge diameter
      * `url`(optional) `null|string`[default: null] URL of the image to display in the badge

  * `minVisibleSize`(optional) `number`[default: 12] If the node diameter on screen is less than this value, the badge will not be displayed
  * `positionScale`(optional) `number`[default: 1] Center of the badge relative to the node size (1 = at the node's border)
  * `scale`(optional) `number`[default: 0.45] Size of the badge relative to the node.
  * `scalingMethod`(optional) `ScalingMethod`[default: `scaled``] Indicates if the badge size should be multiplied by the zoom when the badge is displayed.
  * `stroke`(optional) `object`
      * `color`(optional) `Color|"inherit"`[default: "black"] Color of the badge stroke
      * `scalingMethod`(optional) `ScalingMethod`[default: "fixed"] Indicates if the badge width should be multiplied by the zoom when the node is displayed.
      * `width`(optional) `PixelSize`[default: 2] Width of the badge stroke

  * `text`(optional) `object|TextContent` If not an object, alias for `text.content`
      * `color`(optional) `"inherit"|Color`[default: "black"] Color of the badge text
      * `content`(optional) `TextContent`[default: null] Text to display in the badge
      * `font`(optional) `string`[default: "Arial"] Font to use for the badge text
      * `paddingLeft`(optional) `number`[default: 0] Horizontal padding of the text inside the badge
      * `paddingTop`(optional) `number`[default: 0] Vertical padding of the text inside the badge
      * `scale`(optional) `number`[default: 0.5] Size of the text relative to the badge diameter
      * `style`(optional) `FontStyle`[default: "normal"] Style applied to the badge text

## `BadgeText`

**Type:**  `object`

**Properties**
  * `color`(optional) `"inherit"|Color`[default: "black"] Color of the badge text
  * `content`(optional) `TextContent`[default: null] Text to display in the badge
  * `font`(optional) `string`[default: "Arial"] Font to use for the badge text
  * `paddingLeft`(optional) `number`[default: 0] Horizontal padding of the text inside the badge
  * `paddingTop`(optional) `number`[default: 0] Vertical padding of the text inside the badge
  * `scale`(optional) `number`[default: 0.5] Size of the text relative to the badge diameter
  * `style`(optional) `FontStyle`[default: "normal"] Style applied to the badge text

## `BaseMapOptions`

**Type:**  `object`

Options for the base layer of the map in geo mode.

**Properties**
  * `BaseMapOptions`(optional)
      * `subdomains`(optional) `string`[default: &#039;abc&#039;] Format of the URL used to search for tiles.
      * `tms`(optional) `boolean`[default: false] Useful when using a `TMS` service.
      * `url`(optional) `string` Format of the URL used to search for tiles. Must contain `'{z}'` (zoom level of the tile), `'{x}'` and `'{y}'` (coordinates of the tile). Possible `'{s}'` (replaced by one of the characters in the `tileUrlSubdomains` setting). Possible `'{r}'` for tile servers who support retina tiles.
      * `wms`(optional) `boolean`[default: false] Useful when using a `WMS` service. For custom WMS parameters the Leaflet WMS TileLayer
                                               options can be used as reference for additional parameters to be set

## `BetweennessOptions`

**Type:**  `object`

**Properties**
  * `directed `(optional) `boolean`[default: false] If true, the algorithm will take account of the edge directions.
  * `edges`(optional) `EdgeList<ED, ND>` Edges to take in account while computing the betweenness. All visible edges by default.
  * `nodes`(optional) `NodeList<ND,ED>` Nodes to compute the betweenness on. All visible nodes by default.
  * `normalized`(optional) `boolean`[default: true] If true, will normalize the values within [0;1]

## `BoundingBox`

**Type:**  `object`

**Properties**
  * `computeForZoom` `function(zoom: number):BoundingBox` Method to include the maximum item
sizes to compute the real boundaries at the selected zoom level
  * `cx` `number` X coordinate of the center of the bounding box
  * `cy` `number` Y coordinate of the center of the bounding box
  * `height` `number` Height of the bounding box
  * `maxFixedSize` `number` Maximum fixed size of the elements used to compute the bounding box
  * `maxScaledSize` `number` Maximum scaled size of the elements used to compute the bounding box
  * `maxX` `number` Maximum X coordinate of the bounding box
  * `maxY` `number` Maximum Y coordinate of the bounding box
  * `minFixedSize` `number` Minimum fixed size of the elements used to compute the bounding box
  * `minScaledSize` `number` Minimum scaled size of the elements used to compute the bounding box
  * `minX` `number` Minimum X coordinate of the bounding box
  * `minY` `number` Minimum Y coordinate of the bounding box
  * `pad` `function(value: number):BoundingBox` Extends the bounding box by value on every direction
  * `width` `number` Width of the bounding box

## `BrandOptions`

**Type:**  `object`

**Properties**
  * `className`(optional) `string` If specified, this class will be added to the HTML created HTML element.
  * `horizontalMargin`(optional) `number`[default: 0] Indicates the space in pixels between the brand and the top/bottom of the screen (depending on the position)
  * `position`(optional) `"top-left"|"top-right"|"bottom-left"|"bottom-right"`[default: "bottom-right"] Indicates the position of the brand.
  * `verticalMargin`(optional) `number`[default: 0] Indicates the space in pixels between the brand and the right/left of the screen (depending on the position)

## `CameraAnimationOptions`

**Type:**  `object`

**Properties**
  * `duration`(optional) `number`[default: 0] Duration of the animation, in milliseconds.
  * `easing`(optional) `EasingFunction`[default: "linear"] Easing used by the animation.
  * `ignoreZoomLimits`(optional) `boolean`[default: false] If `false`, the options `interactions.zoom.minValue` and `interactions.zoom.maxValue` are ignored.
  * `startAfter`(optional) `number`[default: 0] Advanced usage. Number from 0 to 1 indicating after which percentage of the duration the animation must be started. For example,
                                  specifying 0.5 would cause the animation to start after half of its total duration, and to be played
                                  two times faster as a consequence.

## `CanvasLayer`

**Type:**  `object`

The layer object containing its properties (overloaded for handling canvas).

**Properties**
  * `destroy` `() => CanvasLayer` Destroy the layer (remove it from the layer array, remove its listeners).
  * `element` `HTMLElement` HTML element used by the layer.
  * `getLevel` `() => number` Retrieves the index of the layer in the layer array.
  * `getOpacity` `() => number` Get the layer opacity, between 0 and 1.
  * `hide` `() => CanvasLayer` Keep the layer in the layer array but hide its content.
  * `isHidden` `() => boolean` Check the layer visibility.
  * `moveDown` `() => CanvasLayer` Take the layer down a notch (decreases its index by 1).
  * `moveTo` `(depth: number) => CanvasLayer` Move the layer to the specified index in the layer array.
  * `moveToBottom` `() => CanvasLayer` Move the layer at the very bottom of the array (index `0`).
  * `moveToTop` `() => CanvasLayer` Move the layer at the very top of the array (index `length - 1`).
  * `moveUp` `() => CanvasLayer` Take the layer up a notch (increases its index by 1).
  * `refresh` `DrawingFunction` Function rerendering the canvas.
  * `setOpacity` `(opacity: number) => CanvasLayer` Set the layer opacity.
  * `show` `() => CanvasLayer` Show the layer content.

## `CanvasLayerOptions`

**Type:**  `object`

Canvas layer options.

**Properties**
  * `isStatic`(optional) `boolean`[default: false] Is the canvas sync with the view ?
  * `noClear`(optional) `boolean`[default: false] No clearing is executed before each draw call.

## `Color`

**Type:**  `null|string`

CSS color name (e.g `"red"`), `"transparent"`,
rgb(a) notation (e.g: `"rgb(128, 128, 128)"` or `"rgba(128, 128, 128, 0.2)"`),
hexadecimal notation (e.g: `"#FFFFFF"`) or `null` (transparent).

## `CrossOriginValue`

**Type:**  `"anonymous"|"use-credentials"|null`

## `CursorStyle`

**Type:**  `string`

CSS value which will be assigned to the cursor.
List of available values here.

## `Dependency`

**Type:**  `object|true`

`true` indicates that the rule/class
depends on the existence of the node(s)/edge(s), but not on their attributes

**Properties**
  * `attributes`(optional) `"all"|Array<PropertyPath>` List of attributes the
rule/class depends on
  * `data`(optional) `boolean` Indicates if the result of the rule is based on
the node/edge's data
  * `hover`(optional) `boolean` Indicates if the result of the rule changes
depending on whether the node/edge is hovered or not
  * `selection`(optional) `boolean` Indicates if the result of the rule changes
depending on whether the node/edge is selected or not

## `DrawingFunction`

**Type:**  `(ctx: CanvasRenderingContext2D) => void`

The function drawing on the canvas in the graph space.

## `DrillDownOptions`

**Type:**  `object`

**Properties**
  * `copyData`(optional) `boolean`[default: true] If true, the data of the open node will be copied into the group replacing it.
  * `depthPath`(optional) `string`[default: "depth"] Data path in the node that contains the depth of the node.
  * `duration`(optional) `number` Animation time in milliseconds. The animation will be played once after the creation of the group. Ignored if `enabled` is false.
  * `easing`(optional) `EasingFunction`[default: "quadraticOut"] Grouping animation easing function.
  * `nodeGenerator`(optional) `function(nodes: NodeList, groupId: string, transformation: Transformation): NodeDataAndAttributes` See nodeGrouping [nodeGenerator]{@link types/nodegroupingoptions}.
  * `onGetSubgraph` `function(node: Node): Promise<RawGraph>` Function that should return a promise that resolves with the subgraph inside the group.
  * `onGroupUpdate`(optional) `function(metaNode: Node, subNodes: NodeList, isOpen: boolean): SubLayout` Callback to run a layout over the content of an open group.
It should return a promise that resolves when the layout is done, or a Promise containing an array of positions for each node in the group.
  * `padding`(optional) `number|function(metaNode: Node, depth: number): number` Padding applied to groups showing their content. If a function is passed, each MetaNode will get a padding depending on that function.
  * `parentPath`(optional) `string`[default: "parent"] Data path in the node that contains the id of the parent node.
  * `showContents`(optional) `function(metaNode: Node, depth: number): boolean | boolean` Function that would define if the contents of the group node should be hidden or not. This is called on every transformation update.

## `Easing`

**Type:**  `"linear"|"quadraticIn"|"quadraticOut"|"quadraticInOut"|"cubicIn"|"cubicOut"|"cubicInOut"`

## `EasingFunction`

**Type:**  `"linear"|"quadraticIn"|"quadraticOut"|"quadraticInOut"|"cubicIn"|"cubicOut"|"cubicInOut"|function(x: number): number`

## `EdgeAttributesValue`

**Type:**  `object`

Object following the same structure as
`EdgeAttributes`, with the addition that each property can
be replaced by a function that return a value for this property (or an object
if the property has multiple nested sub-properties).
When working with a large number of nodes/edges, avoid as much as possible the usage of functions.

## `EdgeCollection`

**Type:**  `Edge|EdgeList|EdgeId|Array<Edge|EdgeId>`

## `EdgeDataAndAttributes`

**Type:**  `object`

**Properties**
  * `attributes`(optional) `EdgeAttributes`
  * `data`(optional) `any`
  * `id`(optional) `EdgeId`

## `EdgeDependencies`

**Type:**  `object|null`

If `null`, indicates that the edge attributes defined by the rule/class does
not depend on any attribute of any node/edge.
If unspecified, the `self`, `extremities` and `parallelEdges` fields are
treated as `{attributes: "all", data: true, selection: true, hover: false}`,
and the `allNodes` and `allEdges` fields are treated as `null`.

**Properties**
  * `allEdges`(optional) `Dependency` Indicates that the rule/class for that
edge should be updated when the specified attributes of any edge change
  * `allNodes`(optional) `Dependency` Indicates that the rule/class for that
edge should be updated when the specified attributes of any node change
  * `extremities`(optional) `Dependency` Indicates that the rule/class for that
edge should be updated when the specified attributes of the edge's
extremities change
  * `parallelEdges`(optional) `Dependency` Indicates that the rule/class for
that edge should be updated when the specified attributes of the node's
parallel edges change
  * `self`(optional) `Dependency` Indicates that the rule/class for that edge
should be updated when the specified attributes of the edge change

## `EdgeDirection`

**Type:**  `"both"|"in"|"out"`

## `EdgeExtremity`

**Type:**  `null|"arrow"|"circle-hole-arrow"|"triangle-hole-arrow"|"short-arrow"|"sharp-arrow"|"circle"|"square"`

## `EdgeFilterOptions`

**Type:**  `object`

**Properties**
  * `options`
      * `criteria` `function(edge: Edge): boolean`
      * `duration`(optional) `number`[default: 0] Duration of the filtering animation. When filtering is animated,
edges are gradually faded, before disappearing completely. Ignored if `enabled` is false.
      * `enabled`(optional) `boolean`[default: true] Indicates if the filter must be enabled.

  * `easing`(optional) `EasingFunction`[default: "quadraticOut"] Fading animation easing function.

## `EdgeGroupingOptions`

**Type:**  `object`

**Properties**
  * `duration`(optional) `number` Animation time in milliseconds. The animation will be played once after the creation of the group. Ignored if `enabled` is false.
  * `easing`(optional) `EasingFunction`[default: "quadraticOut"] Grouping animation easing function.
  * `enabled`(optional) `boolean` Indicates if the edge grouping must be enabled.
  * `generator`(optional) `function(edges: EdgeList, groupId: string, source: Node, target: Node, transformation: Transformation): EdgeDataAndAttributes| null` Given a list of edges that should be grouped together, must return the new edge (meta-edge) to be added. Returning null means that the edges should not be grouped.
  * `groupIdFunction`(optional) `function(edge: Edge): string|undefined` Given an edge, must return a string identifying a
group. All edges that are parallel and for which the function returns the same value will be grouped together. By
default matches all the edges that are parallel together. The returned string will be used as prefix of the final id. Returning `undefined` means that the edge should not be grouped.
  * `selector`(optional) `function(edge: Edge): boolean` Only edges that match this criteria will be grouped with
other edges. By default, all the edges will be assigned a group.
  * `separateEdgesByDirection`(optional) `boolean` By default, edges that have opposite source and target
are grouped together (resulting source and target not deterministic). If this option is `true`, they will not be grouped together.

## `EdgeId`

**Type:**  `string|number`

## `EdgeOutput`

**Type:**  `object|null`

If unspecified, the assigned attributes are inferred to the best possible
extent from the `EdgeAttributesValue` value.

**Properties**
  * `attributes`(optional) `"all"|Array<PropertyPath>` List of edge attributes
assigned by the rule/class

## `EdgeSelector`

**Type:**  `null|function(edge: Edge): boolean`

Used to indicate if a style rule should be applied to a given edge. `null`
is equivalent to a function that always returns true.

## `EdgeSelector`

**Type:**  `null|function(edge: Edge): boolean`

Used to indicate if a style rule should be applied to a given edge. `null`
is equivalent to a function that always returns true.

## `EdgeStyle`

**Type:**  `"plain"|"dotted"|"dashed"`

## `EdgeTextPosition`

**Type:**  `"shifted"|"centered"`

## `EdgeType`

**Type:**  `"line"|"triangle"`

## `EdgesRoutingStyle`

**Type:**  `'horizontal' | 'vertical' | 'none'`

Routing direction for curved edges

## `Filter`

**Type:**  `"visible"|"raw"|"all"`

Indicates which nodes and edges to take into account.
"visible" refers to visible element, "raw" refers to elements that are not the result of a transformation (the
"original" graph), and "all" refers to all elements, including the non-visible ones.

## `FilterOptionDuringAction`

**Type:**  `object`

## `FontStyle`

**Type:**  `"normal"|"bold"|"italic"`

## `GeoClusteringOptions`

**Type:**  `object`

**Properties**
  * `edgeGenerator`(optional) `function(edges: EdgeList, groupId: string, transformation: Transformation): EdgeDataAndAttributes|null` If `groupEdges` is `true`, indicates the function used to generate the new edges from the sub-edges. Ignored if
`groupEdges` is `false`. If returned `null`, the meta edge will not be created.
  * `enabled`(optional) `boolean` Indicates if the grouping must be enabled.
  * `groupEdges`(optional) `boolean` Indicates if parallel edges that end up having at least one meta-node
extremity should be grouped together (to reduce cluttering).
  * `groupIdFunction`(optional) `function(node: Node): string|undefined` Given a node, must return a string identifying a
group. All nodes for which the function returns the same value will be grouped together. By default group all the
nodes that were selected together.
  * `groupSelfLoopEdges`(optional) `boolean` If true, edges for which the two extremities end up being grouped
into the same node will be displayed as a self-loop edge, instead of not being displayed at all.
  * `nodeGenerator`(optional) `function(nodes: NodeList, groupId: string, transformation: Transformation): NodeDataAndAttributes|null` Given a list of nodes that should be grouped together, must return the new node (meta-node) to be added.
The latitude and longitude of the meta-node will be the average of the latitude and longitude of the nodes.
If `null` is returned, the meta-node will not be created.
  * `onCreated`(optional) `function(metaNode: Node, showContents: boolean, subNodes: NodeList, subEdges: EdgeList): Promise<any>` This is
a callback called after a new group is created. See {@link ./tutorials/grouping/index.html | here} for more details.
  * `radius`(optional) `number` The radius in which a node gets grouped to annother one.
  * `selector`(optional) `function(node: Node): boolean` Only nodes that match this criteria will be grouped with
other nodes. By default, all the nodes will be assigned a group.
  * `separateEdgesByDirection`(optional) `boolean` By default, edges that have opposite source and target
are grouped together (resulting source and target not deterministic). If this option is `true`, they will not be grouped together.

## `GeoCoordinate`

**Type:**  `object`

Geographical coordinates, latitude and longitude.

**Properties**
  * `latitude` `number` Latitude (degrees)
  * `longitude` `number` Longitude (degrees)

## `GeoModeOptions`

**Type:**  `object`

**Properties**
  * `attribution`(optional) `string`[default: &#039;Map data (c) OpenStreetMap contributors&#039;] HTML string that will be displayed on the
                                              corner, indicates the source of the tiles.
  * `attributionOptions`(optional) `BrandOptions` Position and options for the attribution message
  * `backgroundColor`(optional) `Color`[default: &#039;silver&#039;] Color of the map background (color of the missing tiles).
  * `crs`(optional) `CRS`[default: L.CRS.EPSG3857] This parameter controls the Coordinate Reference System used for
                                               the map projection. Usually you do not need to change it, unless
                                               you need custom projections. For more information see the Leaflet CRS documentation.
  * `disableNodeDragging`(optional) `boolean`[default: true] Disable node dragging when the mode is on. Dragging is disabled
                                                 by default, because geo mode is not scale-free.
  * `duration`(optional) `number`[default: 0] Duration of the transition when swapping mode.
  * `latitude`(optional) `number` Latitude of the map center. If none is provided, the center of the nodes with coordinates will be used.
  * `latitudePath`(optional) `PropertyPath`[default: &#039;latitude&#039;] Node path which contains the latitude.
  * `longitude`(optional) `number` Longitude of the map center. If none is provided, the center of the nodes with coordinates will be used.
  * `longitudePath`(optional) `PropertyPath`[default: &#039;longitude&#039;] Node path which contains the longitude.
  * `maxZoomLevel`(optional) `number`[default: 20] Maximum geo-spatial zoom.
  * `minZoomLevel`(optional) `number`[default: 1] Minimum geo-spatial zoom.
  * `opacity`(optional) `number`[default: 1] Map baselayer opacity
  * `sizeRatio`(optional) `number`[default: 1] Multiplier for the node radius an edge width.
  * `tileBuffer`(optional) `number`[default: 2] Number of extra tiles to be downloaded around
                                              the viewport bounds, to make the panning smoother.
                                              Deprecated: this is automatically handled now, alternatively,
                                              you can pass this to the `tiles` together with the `L.TileLayer`
                                              options.
  * `tileUrlSubdomains`(optional) `string`[default: &#039;abc&#039;] Values with which the '{s}' string in the URL can be replaced.
                                              Deprecated, use `tiles.subdomains` instead.
  * `tileUrlTemplate`(optional) `string`[default: &#039;https://{s}.tile.osm.org/{z}/{x}/{y}.png&#039;] Format of the URL used to search for tiles.
                                              Must contain `'{z}'` (zoom level of the tile),
                                              `'{x}'` and `'{y}'` (coordinates of the tile).
                                              Possible `'{s}'` (replaced by one of the
                                              characters in the `tileUrlSubdomains` setting).
                                              Possible `'{r}'` for tile servers who support retina tiles.
                                              Deprecated: use `tiles.url` now.
  * `tiles`(optional) `object|L.Layer` In addition to the following list, this object supports also Leaflet TileLayer Options
                                              and Leaflet WMS TileLayer Options options
      * `subdomains`(optional) `string`[default: &#039;abc&#039;] Format of the URL used to search for tiles.
      * `tms`(optional) `boolean`[default: false] Useful when using a `TMS` service.
      * `url`(optional) `string`[default: &#039;https://{s}.tile.osm.org/{z}/{x}/{y}.png&#039;] Format of the URL used to search for tiles.
                                              Must contain `'{z}'` (zoom level of the tile),
                                              `'{x}'` and `'{y}'` (coordinates of the tile).
                                              Possible `'{s}'` (replaced by one of the
                                              characters in the `tileUrlSubdomains` setting).
                                              Possible `'{r}'` for tile servers who support retina tiles.
      * `wms`(optional) `boolean`[default: false] Useful when using a `WMS` service. For custom WMS parameters the Leaflet WMS TileLayer
                                               options can be used as reference for additional parameters to be set

  * `wrapCoordinates`(optional) `boolean`[default: true] Whether to wrap the coordinate to the projection space of
                                              [-180, 180] for longitude, [-85, 85] for longitude.
                                              If set to false, nodes with coordinates outside of that
                                              range will not be shown.
  * `zoom`(optional) `number` Zoom level of the map. If none is provided, the zoom level that fits all nodes with coordinates will be used.

## `GroupLayout`

**Type:**  `object`

**Properties**
  * `groupId` `NodeId` The id of the group to layout.
  * `layout` `string` The name of the layout to apply to the group (force, hierarchical etc).
  * `params` `object` The parametters of the layout to apply.

## `HoverEdgeOptions`

**Type:**  `object`

Options for the hovered edge.
(EdgeAttributesValue +
AnimationOptions)
You can specify the duration and easing of the hover animation here.

## `HoverNodeOptions`

**Type:**  `object`

Options for the hovered node.
(NodeAttributesValue +
AnimationOptions)
You can specify the duration and easing of the hover animation here.

## `ImageExportOptions`

**Type:**  `object`

**Properties**
  * `background`(optional) `Color` Color of the background on the output image
  * `badges`(optional) `boolean`[default: true] Whether or not to export badges
  * `clip`(optional) `boolean`[default: false] If `true`, export the current view rather than the whole graph
  * `download`(optional) `boolean`[default: true] If true, the user will be prompted a modal window so he can download the exported graph.
  * `filename`(optional) `string`[default: "graph.png"] If `download` is true, the default name for the downloaded file.
  * `filter`(optional) `"visible"|"all"`[default: "visible"] Indicates what elements to export.
  * `height`(optional) `number` If not specified, the height of the canvas will be used.
  * `imageCrossOrigin`(optional) `string` In case node or badge images are coming from a CDN, set this to
                                                             'anonymous' to avoid security errors on export.
  * `imageWatermark`(optional) `object`
      * `alpha`(optional) `string`[default: 0.65] Transparency of the watermark, from 0 to 1 (0 = fully transparent)
      * `angle`(optional) `string`[default: 0] Angle of the watermark (in degrees)
      * `height`(optional) `string` If not specified, the original width of the image will be used.
      * `repeat`(optional) `string`[default: false] Indicates if the watermark should be repeated over the image
      * `space`(optional) `string`[default: 50] If repeating the watermark, space in pixels between the repetitions
      * `url`(optional) `string` Url of the image to use as watermark
      * `width`(optional) `string` If not specified, the original width of the image will be used.
      * `x`(optional) `string` X coordinate of the center of the watermark
      * `y`(optional) `string` Y coordinate of the center of the watermark

  * `images`(optional) `boolean`[default: true] Whether or not to export images
  * `layers `(optional) `boolean`[default: true] Whether to export visible layers or not. Defaults to true
  * `legend`(optional) `boolean|LegendOptions` If unspecified and the legend is enabled, it will be exported with the current options.
If unspecified and the legend is not enabled, it will not be exported. If `false`, the legend will not be exported no matter what.
If `true`, the legend will be exported with the default options, whether it's enabled or not. If an object is specified,
the legend will be exported with the specified options, whether it's enabled or not.
  * `margin`(optional) `number`[default: 10] Blank space around the graph (in pixels)
  * `pixelRatio`(optional) `number` Image resolution. Defaults to your screen resolution, so on high-resolution screens the exported image will be scaled up to maintain the resolution.
  * `preventOverlap`(optional) `boolean`[default: true] If true, hide texts which overlap, else display labels which are already present on screen
  * `textWatermark`(optional) `object`
      * `alpha`(optional) `number`[default: 0.65] Transparency of the watermark, from 0 to 1 (0 = fully transparent)
      * `angle`(optional) `number`[default: 0] Angle of the watermark (in degrees)
      * `content`(optional) `string` Content of the text watermark
      * `fontColor`(optional) `Color`[default: "red"] Color to use to display the text watermark
      * `fontFamily`(optional) `string`[default: "Arial"] Font used to display the text watermark
      * `fontSize`(optional) `number`[default: 48] Size of the text watermark
      * `fontStyle`(optional) `"bold"|"italic"` Style to use to display the text watermark
      * `repeat`(optional) `boolean`[default: false] Indicates if the watermark should be repeated over the image
      * `space`(optional) `number`[default: 50] If repeating the watermark, space in pixels between the repetitions
      * `x`(optional) `number` X coordinate of the center of the watermark
      * `y`(optional) `number` Y coordinate of the center of the watermark

  * `texts`(optional) `boolean`[default: true] Whether or not to export texts
  * `width`(optional) `number` If not specified, the width of the canvas will be used.

## `InputSource`

**Type:**  `"mouse"|"touch"`

Indicates what kind of source emitted the event.

## `InputTarget`

**Type:**  `Node|Edge|null`

Element that is clicked/hovered. If it's not `null`, you can check
if it's a node or an edge with their `isNode` property.

## `InteractionOptions`

**Type:**  `object`

## `KeyCode`

**Type:**  `number`

JavaScript key code.

## `KeyName`

**Type:**  `string`

Lowercase letter (e.g `"a"`), digit (e.g `"3"`) or
`"shift"|"ctrl"|"cmd"|"alt"|"space"|"enter"|"esc"|"del"|"backspace"`.

## `Layer`

**Type:**  `object`

The layer object containing its properties.

**Properties**
  * `destroy` `() => Layer` Destroy the layer (remove it from the layer array, remove its listeners).
  * `element` `HTMLElement` HTML element used by the layer.
  * `getLevel` `() => number` Retrieves the index of the layer in the layer array.
  * `getOpacity` `() => number` Get the layer opacity, between 0 and 1.
  * `hide` `() => Layer` Keep the layer in the layer array but hide its content.
  * `isHidden` `() => boolean` Check the layer visibility.
  * `moveDown` `() => Layer` Take the layer down a notch (decreases its index by 1).
  * `moveTo` `(depth: number) => Layer` Move the layer to the specified index in the layer array.
  * `moveToBottom` `() => Layer` Move the layer at the very bottom of the array (index `0`).
  * `moveToTop` `() => Layer` Move the layer at the very top of the array (index `length - 1`).
  * `moveUp` `() => Layer` Take the layer up a notch (increases its index by 1).
  * `setOpacity` `(opacity: number) => Layer` Set the layer opacity.
  * `setPosition` `(position: {x: number, y: number}) => Layer` Setter setting the element translation in the graph space.
  * `setSize` `(size: {width: number, height: number}) => Layer` Setter setting the element size in the graph space.
  * `show` `() => Layer` Show the layer content.

## `LayerValue`

**Type:**  `number`

Integer between -1 and 3 (included). By default elements are on the layer 0.
Selected elements are on the layer 2. Hovered elements are on the layer 3.

## `LayoutOptions`

**Type:**  `object`

**Properties**
  * `continuous`(optional) `boolean`[default: false] Whether or not the layout should render intermediate steps.
  * `dryRun`(optional) `boolean`[default: false] If true, the layout will not be applied to the graph.
  * `duration`(optional) `number`[default: 300] Duration of the animation when the graph is updated
  * `easing`(optional) `Easing`[default: &#039;cubicIn&#039;] Easing function used during the animation
  * `locate`(optional) `boolean|LocateOptions`[default: false] Center on the graph bounding box when the layout is complete. You can also provide padding.
  * `nodes`(optional) `NodeId[]|NodeList` List of affected nodes. If nothing provided, the whole graph will be used. Where edges param is available and provided, then this list will be augmented with reached nodes from the passed edge list.
  * `onEnd`(optional) `function(): void` Function called after the last graph update
  * `onSync`(optional) `function(): void` Function called every time the graph is updated
  * `skipTextDrawing`(optional) `boolean`[default: true] Skip drawing labels during the layout. Improves performance and user experience.
  * `useWebWorker`(optional) `boolean`[default: true] Indicates if the layout should be computed inside a web worker.

## `LegendOptions`

**Type:**  `object`

**Properties**
  * `backgroundColor`(optional) `Color`[default: "white"] Background color of the widgets.
  * `borderColor`(optional) `Color`[default: "black"] Border color of the widgets.
  * `borderRadius`(optional) `number`[default: 0] Border radius of the widgets.
  * `borderWidth`(optional) `number`[default: 1] Border width of the widgets, in pixels.
  * `circleStrokeWidth`(optional) `number`[default: 3] Stroke width of the circles used to indicate the size of the nodes.
  * `fontColor`(optional) `Color`[default: "black"] Font color used to display the widgets' content
  * `fontFamily`(optional) `string`[default: "Arial"] Font used to display the widgets
  * `fontSize`(optional) `number`[default: 10] Font size used to display the widgets' content
  * `innerMargin`(optional) `number`[default: 10] Blank space between a widget's border and its content, in pixels.
  * `outerMargin`(optional) `number`[default: 5] Blank space between two widgets, in pixels.
  * `position`(optional) `"bottom"|"top"|"left"|"right"`[default: "bottom"] Position of the legend on the canvas.
  * `shapeColor`(optional) `Color`[default: "grey"] Color used for displaying the widget indicating a node or edge shape
  * `titleFontColor`(optional) `Color`[default: "black"] Font color used to display the widgets' title
  * `titleFontSize`(optional) `number`[default: 12] Font size used to display the widgets' title
  * `titleFunction`(optional) `function(propertyPath: Array<string>, styleProperty: PropertyPath): string` Given a property path, must return
the title of the widget which displays information on that property. By default keep the last part of the
property path.
  * `titleMaxLength`(optional) `number`[default: 20] If a widget's title has more characters that this value, it will be truncated
  * `titleTextAlign`(optional) `"left"|"center"`[default: "left"] Alignment of the widgets' title
  * `widgetWidth`(optional) `number`[default: 130] Width of a widget, in pixels

## `LocateOptions`

**Type:**  `object`

**Properties**
  * `duration`(optional) `number`[default: 0] Duration of the camera movement, in milliseconds.
  * `easing`(optional) `EasingFunction`[default: "cubicOut"] Easing function applied to the movement of the camera.
  * `ignoreZoomLimits`(optional) `boolean`[default: false] If `true`, the options `interactions.zoom.minValue` and
                                                         `interactions.zoom.maxValue` are ignored.
  * `maxNodeSizeOnScreen`(optional) `number`[default: 200] Additional restriction on the zoom that makes sure no node
                                                         is displayed with a diameter greater than this value, in pixels.
                                                         Set to 0 to disable this feature.
  * `padding`(optional) `object|number` If a number, indicates the padding for the four sides.
      * `bottom`(optional) `number`[default: 40] Bottom padding (in pixels).
      * `left`(optional) `number`[default: 40] Left padding (in pixels).
      * `right`(optional) `number`[default: 40] Right padding (in pixels).
      * `top`(optional) `number`[default: 40] Top padding (in pixels).

## `MapPosition`

**Type:**  `object`

Geo coordinates and zoom level of the map

**Properties**
  * `latitude` `number` Latitude (degrees)
  * `longitude` `number` Longitude (degrees)
  * `zoom` `number` Map scale

## `MouseButton`

**Type:**  `"left"|"right"|"middle"`

Identifies a mouse button.

## `NeighborGenerationOptions`

**Type:**  `object`

**Properties**
  * `options`(optional)
      * `duration`(optional) `number`[default: 0]
      * `edgeGenerator`(optional) `function(originalNode: Node, generatedNode: Node): RawEdge` Given the original node and the generated neighbor, returns the edge that must be created between them.
Source and target may be specified; otherwise the original node is the source and the generated node is the
target. If this parameter is unspecified, an edge with no data and with default attributes is created.
      * `enabled`(optional) `boolean`[default: true]
      * `neighborIdFunction` `function(node: Node): string|Array<string>|null` String identifying the neighbor that will be created. An array of string can be passed; in this case multiple
neighbors are created. If `null` is returned, no neighbor is created.
      * `nodeGenerator`(optional) `function(identifier: string, nodes: NodeList): RawNode` Given the nodes that spawned it and its identifier (which is not its final id, but is part of it),
returns the node that must be created. If unspecified, a node with no data and with default attributes
is created.
      * `selector`(optional) `function(node: Node): boolean` If unspecified, defaults to a function that
always returns true.

  * `easing`(optional) `EasingFunction`[default: "quadraticOut"] Animation easing function.

## `NeighborMergingOptions`

**Type:**  `object`

**Properties**
  * `options`
      * `dataFunction`(optional) `function(node: Node): object|undefined` If unspecified or if it returns
`undefined`, no data is added to the neighbors.
      * `selector` `function(node: Node): boolean` Function indicating which nodes will be merged.

  * `duration`(optional) `number`[default: 0]
  * `easing`(optional) `EasingFunction`[default: "quadraticOut"] Animation easing function.
  * `enabled`(optional) `boolean`[default: true]

## `NodeAttributesValue`

**Type:**  `object`

Object following the same structure as `NodeAttributes`,
with the addition that each property can be replaced by a function that return a value for this property (or an object
if the property has multiple nested sub-properties). `undefined` can be explicitly specified in any field to indicate that
the attribute should not be modified (useful when updating a class/rule).
When working with a large number of nodes/edges, avoid as much as possible the usage of functions.

## `NodeClusteringOptions`

**Type:**  `object`

**Properties**
  * `options`(optional)
      * `edgeGenerator`(optional) `function(edges: EdgeList, clusterId: string, transformation: Transformation): EdgeDataAndAttributes` If `groupEdges` is `true`, indicates the function used to generate the new edges from the sub-edges. Ignored if
`clusterEdges` is `false`.
      * `enabled`(optional) `boolean`[default: true] Indicates if the clustering must be enabled.
      * `groupIdFunction`(optional) `function(node: RawNode): string` Given a node, must return a string identifying a
cluster. All nodes for which the function returns the same value will be clustered together. By default cluster all the
nodes that were selected together.
      * `groupSelfLoopEdges`(optional) `boolean`[default: false] If true, edges for which the two extremities end up being clustered
into the same node will be displayed as a self-loop edge, instead of not being displayed at all.
      * `nodeGenerator`(optional) `function(nodes: NodeList, clusterId: string, transformation: Transformation): NodeDataAndAttributes` Given a list of nodes that should be clustered together, must return the new node (meta-node) to be added.
If `attributes.x` and `attributes.y` are not specified, the meta-node will be put at the center of the nodes.
      * `selector`(optional) `function(node: RawNode): boolean` Only nodes that match this criteria will be clustered with
other nodes. By default, all the nodes will be assigned a cluster.

## `NodeCollapsingOptions`

**Type:**  `object`

**Properties**
  * `options`
      * `duration`(optional) `number`[default: 0]
      * `edgeGenerator`(optional) `function(hiddenNode: Node, node1: Node, node2: Node, edges1: EdgeList, edges2: EdgeList): RawEdge|null` Function returning the RawEdge to create for each pair of adjacent nodes.
`hiddenNode` is the node that is hidden (matches `selector`).
`node1` is the first neighbor of the pair.
`node2` is the second neighbor of the pair.
`edges1` is the list of edges between the hidden node and the first neighbor of the pair.
`edges2` is the list of edges between the hidden node and the second neighbor of the pair.
The source and target of the edge may be specified; if it's the case they must be either `node1` or `node2`.
If it's not the case, an exception will be thrown. If source/target are not specified and the edges follow a
consistent direction (e.g A <- B <- C), the source and target are assigned according to that direction
(here A <- B). Otherwise (e.g A -> B <- C), `node1` is used as the source and `node2` is used as the target.
If `null` is returned, no edge is created between the two nodes.
If no edge generator is specified, a default edge is created.
      * `enabled`(optional) `boolean`[default: true]
      * `selector` `function(node: Node): boolean` Function indicating which nodes will be hidden.

  * `easing`(optional) `EasingFunction`[default: "quadraticOut"] Animation easing function.

## `NodeCollection`

**Type:**  `Node|NodeList|NodeId|Array<Node|NodeId>`

## `NodeDataAndAttributes`

**Type:**  `object`

**Properties**
  * `attributes`(optional) `NodeAttributes`
  * `data`(optional) `any`
  * `id`(optional) `NodeId`

## `NodeDependencies`

**Type:**  `object|null`

If `null`, indicates that the node attributes defined by the rule/class does
not depend on any attribute of any node/edge.
If unspecified, the `self`, `adjacentNodes` and `adjacentEdges` fields are
treated as `{attributes: "all", data: true, selection: true, hover: false}`,
and the `allNodes` and `allEdges` fields are treated as `null`.

**Properties**
  * `adjacentEdges`(optional) `Dependency` Indicates that the rule/class for that
node should be updated when the specified attributes of the node's adjacent
edges change
  * `adjacentNodes`(optional) `Dependency` Indicates that the rule/class for that
node should be updated when the specified attributes of the node's adjacent
nodes change
  * `allEdges`(optional) `Dependency` Indicates that the rule/class for that node
should be updated when the specified attributes of any edge change
  * `allNodes`(optional) `Dependency` Indicates that the rule/class for that node
should be updated when the specified attributes of any node change
  * `self`(optional) `Dependency` Indicates that the rule/class for that node
should be updated when the specified attributes of the node change

## `NodeFilterOptions`

**Type:**  `object`

**Properties**
  * `options`
      * `criteria` `function(node: Node): boolean`
      * `duration`(optional) `number`[default: 0] Duration of the filtering animation. When filtering is animated,
nodes are gradually faded, before disappearing completely. Ignored if `enabled` is false.
      * `enabled`(optional) `boolean`[default: true] Indicates if the filter must be enabled.

  * `easing`(optional) `EasingFunction`[default: "quadraticOut"] Fading animation easing function.

## `NodeGroupingOptions`

**Type:**  `object`

**Properties**
  * `duration`(optional) `number` Animation time in milliseconds. The animation will be played once after the creation of the group. Ignored if `enabled` is false.
  * `easing`(optional) `EasingFunction`[default: "quadraticOut"] Grouping animation easing function.
  * `edgeGenerator`(optional) `function(edges: EdgeList, groupId: string, source: Node, target: Node, transformation: Transformation): EdgeDataAndAttributes` If `groupEdges` is `true`, indicates the function used to generate the new edges from the sub-edges. Ignored if
`groupEdges` is `false`.
  * `enabled`(optional) `boolean` Indicates if the grouping must be enabled.
  * `groupEdges`(optional) `boolean` Indicates if parallel edges that end up having at least one meta-node
extremity should be grouped together (to reduce cluttering).
  * `groupIdFunction`(optional) `function(node: Node): string|undefined` Given a node, must return a string identifying a
group. All nodes for which the function returns the same value will be grouped together. By default group all the
nodes that were selected together. Returning `undefined` means that the node should not be grouped.
  * `groupSelfLoopEdges`(optional) `boolean` If true, edges for which the two extremities end up being grouped
into the same node will be displayed as a self-loop edge, instead of not being displayed at all.
  * `nodeGenerator`(optional) `function(nodes: NodeList, groupId: string, transformation: Transformation): NodeDataAndAttributes` Given a list of nodes that should be grouped together, must return the new node (meta-node) to be added.
GeoClustering will compute automatically the coordinate of the clusters.
  * `onGroupUpdate`(optional) `function(metaNode: Node, subNodes: NodeList, isOpen: boolean): SubLayout` This is a callback allowing you to layout the content of an open group.
It should return a promise that resolves when the layout is done, or a Promise containing an array of positions for each node in the group.
  * `padding`(optional) `number|function(metaNode: Node): number` Padding applied to groups showing their content. If a function is passed, each MetaNode will get a padding depending on that function.
  * `restorePositions`(optional) `boolean` When the grouping is disabled/destroyed, indicates if the nodes should be positioned back around the meta-node position
  * `selector`(optional) `function(node: Node): boolean` Only nodes that match this criteria will be grouped with
other nodes. By default, all the nodes will be assigned a group.
  * `separateEdgesByDirection`(optional) `boolean` By default, edges that have opposite source and target
are grouped together (resulting source and target not deterministic). If this option is `true`, they will not be grouped together.
  * `showContents`(optional) `function(metaNode: Node): boolean | boolean` Function that would define if the contents of the group node should be hidden or not. This is called on every transformation update.

## `NodeId`

**Type:**  `string|number`

## `NodeOutput`

**Type:**  `object|null`

If unspecified, the assigned attributes are inferred to the best possible
extent from the `NodeAttributesValue` value.

**Properties**
  * `attributes`(optional) `"all"|Array<PropertyPath>` List of node attributes
assigned by the rule/class

## `NodeSelector`

**Type:**  `null|function(node: Node): boolean`

Used to indicate if a style rule should be applied to a given node. `null`
is equivalent to a function that always returns true.

## `NodeSelector`

**Type:**  `null|function(node: Node): boolean`

Used to indicate if a style rule should be applied to a given node. `null`
is equivalent to a function that always returns true.

## `NodeShape`

**Type:**  `"circle"|"cross"|"diamond"|"pentagon"|"square"|"star"|"equilateral"`

## `OpacityValue`

**Type:**  `number`

Value between 0 (transparent) and 1 (opaque) indicating the opacity of the node/edge.
Note that Ogma doesn't perform real opacity, but background blending: the lower the opacity value is, the more
the color of the node/edge is blended towards the background color, but it retains its original alpha value.
This has one important implication: when using a transparent background and an image is displayed behind Ogma,
it is necessary to set the RGB values to the background color that are close to the image, even if the color is
transparent. For example, assuming the image is mainly grey, you should do
`ogma.setOptions({backgroundColor: "rgba(128, 128, 128, 0)"})`
so the nodes/edges are nicely blended towards the image color.

## `Options`

**Type:**

**Properties**
  * `backgroundColor`(optional) `Color`[default: "white"] Background color of the canvas.
  * `cursor`(optional) `object`
      * `default`(optional) `CursorStyle`[default: "default"] A CSS value for the cursor.
      * `edge`(optional) `CursorStyle`[default: "pointer"] Cursor style when an edge is hovered.
      * `node`(optional) `CursorStyle`[default: "pointer"] Cursor style when a node is hovered.

  * `detect`(optional) `object`
      * `edgeErrorMargin`(optional) `number`[default: 2] How far of the mouse an edge can be and still be detected, in pixels.
      * `edgeTexts`(optional) `boolean`[default: true] Indicates if the detection of edge texts should be enabled.
      * `edges`(optional) `boolean`[default: true] Indicates if the detection of edges should be enabled.
      * `nodeBadges`(optional) `boolean`[default: true] Indicates if the detection of node badges should be enabled.
      * `nodeErrorMargin`(optional) `number`[default: 5] How far of the mouse a node can be and still be detected, in pixels.
      * `nodeTexts`(optional) `boolean`[default: true] Indicates if the detection of node texts should be enabled.
      * `nodes`(optional) `boolean`[default: true] Indicates if the detection of nodes should be enabled.

  * `directedEdges`(optional) `boolean`[default: false] If true, edges of opposing directions will be not be mixed together visually.
  * `edgeMinTipSize`(optional) `number` The minimum size of the edge tip.
  * `edgeTipRatio`(optional) `number` The ratio of the edge tip size compared to the edge width.
  * `edgesAlwaysCurvy`(optional) `boolean`[default: false] If true, all edges will be curved.
  * `edgesRoutingStyle`(optional) `EdgesRoutingStyle`[default: &#039;none&#039;] If 'horizontal' or 'vertical', cureved edges will have different curvature at the source and target, making horizontal or vertical hierarchies more readable. Use with hierarchical or custom layouts.
  * `imgCrossOrigin`(optional) `CrossOriginValue`[default: "anonymous"] Indicates the value of the `crossOrigin` field for DOM images.
  * `interactions`(optional) `object`
      * `gesture`(optional) `object`
          * `enabled`(optional) `boolean`[default: true] Indicates if zooming/rotating using two fingers should be enabled.
          * `hideEdgeTexts`(optional) `boolean`[default: false] Indicates if the edge texts should be hidden when zooming/rotating the view using two fingers.
          * `hideEdges`(optional) `boolean`[default: false] Indicates if the edges should be hidden when zooming/rotating the view using two fingers.
          * `hideNodeTexts`(optional) `boolean`[default: false] Indicates if the node texts should be hidden when zooming/rotating the view using two fingers.
          * `hideNodes`(optional) `boolean`[default: false] Indicates if the nodes should be hidden when zooming/rotating the view using two fingers.

      * `drag`(optional) `object`
          * `cursor`(optional) `CursorStyle`[default: &#039;move&#039;] Cursor style to be applied while dragging the node
          * `enabled`(optional) `boolean`[default: true] Indicates if dragging nodes with the mouse should be enabled.

      * `pan`(optional) `object`
          * `enabled`(optional) `boolean`[default: true] Indicates if moving the view with the mouse should be enabled.
          * `hideEdgeTexts`(optional) `boolean`[default: false] Indicates if the edge texts should be hidden when moving the view.
          * `hideEdges`(optional) `boolean`[default: false] Indicates if the edges should be hidden when moving the view.
          * `hideNodeTexts`(optional) `boolean`[default: false] Indicates if the node texts should be hidden when moving the view.
          * `hideNodes`(optional) `boolean`[default: false] Indicates if the nodes should be hidden when moving the view.
          * `keyboard`(optional) `boolean`[default: true] Indicates if the keyboard buttons (`↑↓←→`) for moving the view should be enabled.

      * `rotation`(optional) `object`
          * `enabled`(optional) `boolean`[default: true] Indicates if rotating the view with the mouse should be enabled.
          * `hideEdgeTexts`(optional) `boolean`[default: false] Indicates if the edge texts should be hidden when rotating the view.
          * `hideEdges`(optional) `boolean`[default: false] Indicates if the edges should be hidden when rotating the view.
          * `hideNodeTexts`(optional) `boolean`[default: false] Indicates if the node texts should be hidden when rotating the view.
          * `hideNodes`(optional) `boolean`[default: false] Indicates if the nodes should be hidden when rotating the view.

      * `selection`(optional) `object`
          * `enabled`(optional) `boolean`[default: true] Indicates if selection with the mouse should be enabled
          * `multiSelectionKey`(optional) `KeyName|null`[default: "ctrl"] Indicates the key that must be pressed to select multiple nodes/edges at a time.

      * `zoom`(optional) `object`
          * `duration`(optional) `number`[default: 150] Indicate the duration of a manual zoom.
          * `easing`(optional) `EasingFunction`[default: "cubicOut"] Easing function to use for the zoom.
          * `enabled`(optional) `boolean`[default: true] Indicates if zoom on mouse wheel should be enabled.
          * `hideEdgeTexts`(optional) `boolean`[default: false] Indicates if the edge texts should be hidden when
zooming manually.
          * `hideEdges`(optional) `boolean`[default: false] Indicates if the edges should be hidden when zooming
manually.
          * `hideNodeTexts`(optional) `boolean`[default: false] Indicates if the node texts should be hidden when
zooming manually.
          * `hideNodes`(optional) `boolean`[default: false] Indicates if the nodes should be hidden when zooming
manually.
          * `keyboard`(optional) `boolean`[default: true] Indicates if the default `+`/`-` keys are used to zoom in and out.
          * `maxValue`(optional) `null|ZoomBoundaryFunction` Function indicating the maximum possible zoom. By default, it's not possible to zoom so the smallest node takes
more than 50% of the view. Set to `null` to remove the limit.
          * `metaKey`(optional) `'ctrl'|'alt'|'shift'|'space'`[default: &#039;ctrl/cmd&#039;] Modifier button to use for zooming with the
mousewheel or trackpad when `scrollToPan` is enabled. By default it's the `ctrl` (`⌘` on Mac) key.
          * `minValue`(optional) `null|ZoomBoundaryFunction` Function indicating the minimum possible zoom. By default, it's not possible to zoom so the graph takes less
than 20% of the view. Set to `null` to remove the limit.
          * `modifier`(optional) `number`[default: Math.SQRT2] Indicate the zoom multiplier on the manual zoom.
          * `onDoubleClick`(optional) `boolean`[default: false] Indicates if zoom on double click should be enabled.
          * `scrollToPan`(optional) `boolean`[default: false] Indicates if the view should be moved when scrolling with
mousewheel or trackpad
          * `speed`(optional) `number`[default: 0.15] The speed of the zoom on mouse wheel.

  * `minimumHeight`(optional) `number`[default: 300] Minimum container height in pixels.
  * `minimumWidth`(optional) `number`[default: 300] Minimum container width in pixels.
  * `mouse`(optional) `object`
      * `disableWheelUntilMouseDown`(optional) `boolean`[default: false] If true, the canvas will not take the focus of the mouse until the user clicks on it.
      * `doubleClickTimer`(optional) `boolean`[default: 500] After a click, amount of time after which a second click won't trigger a double click event
      * `enabled`(optional) `boolean`[default: true] Indicates if the mouse should be enabled
      * `wheelEnabled`(optional) `boolean`[default: true] Indicates if the mouse wheel should be enabled.

  * `renderer`(optional) `RendererType`[default: "webgl"] Rendering type. If WebGL is selected and not available, Ogma will fallback on Canvas.
If no renderer is available (e.g in Node.js), Ogma will fallback on headless mode (`null`).
  * `texts`(optional) `object`
      * `cornerRadius`(optional) `number`[default: 5] Controls the radius of the rounded corners of text boxes.
Set to 0 to get non rounded corners.
      * `hideUntilFontsLoaded`(optional) `boolean`[default: true] If `true`, texts & icons won't be displayed until the browser
has finished loading the fonts. This prevents texts from showing up with the default font during the short time
between Ogma initialization and the moment fonts are loaded.
      * `preventOverlap`(optional) `boolean`[default: true] Detect and remove overlapping of texts.

  * `touch`(optional) `object`
      * `enabled`(optional) `boolean`[default: true] Indicates if the touch should be enabled

## `Overlay`

**Type:**  `object`

The layer object containing its properties (overloaded for handling transformed elements).

**Properties**
  * `destroy` `() => Overlay` Destroy the layer (remove it from the layer array, remove its listeners).
  * `element` `HTMLElement` HTML element used by the layer.
  * `getLevel` `() => number` Retrieves the index of the layer in the layer array.
  * `getOpacity` `() => number` Get the layer opacity, between 0 and 1.
  * `hide` `() => Overlay` Keep the layer in the layer array but hide its content.
  * `isHidden` `() => boolean` Check the layer visibility.
  * `moveDown` `() => Overlay` Take the layer down a notch (decreases its index by 1).
  * `moveTo` `(depth: number) => Overlay` Move the layer to the specified index in the layer array.
  * `moveToBottom` `() => Overlay` Move the layer at the very bottom of the array (index `0`).
  * `moveToTop` `() => Overlay` Move the layer at the very top of the array (index `length - 1`).
  * `moveUp` `() => Overlay` Take the layer up a notch (increases its index by 1).
  * `setOpacity` `(opacity: number) => Overlay` Set the layer opacity.
  * `setPosition` `(position: {x: number, y: number}) => Overlay` Setter setting the element translation in the graph space.
  * `setSize` `(size: {width: number, height: number}) => Overlay` Setter setting the element size in the graph space.
  * `show` `() => Overlay` Show the layer content.

## `OverlayOptions`

**Type:**  `object`

HTML element provided with its affine transformation in the graph space.

**Properties**
  * `element` `HTMLElement|string` HTML element being transformed. You can also provide an HTML string.
  * `position` `{x: number, y: number}` The element translation in the graph space.
  * `scaled`(optional) `boolean`[default: true] Whether or not the overlay should be scaled together with the graph.
  * `size`(optional) `{width: number, height: number}` The element size in the graph space.

## `Padding`

**Type:**  `object`

**Properties**
  * `bottom` `number` Bottom padding (in pixels).
  * `left` `number` Left padding (in pixels).
  * `right` `number` Right padding (in pixels).
  * `top` `number` Top padding (in pixels).

## `PixelSize`

**Type:**  `number|string`

Indicates a size in pixels. A string in the format `X%` (e.g "200%") can be
specified instead of a number, in which case it is treated as `X percent of
the default value`. `X` should be parsable using `parseFloat`.

## `Point`

**Type:**  `object`

**Properties**
  * `x` `number` X coordinate.
  * `y` `number` Y coordinate.

## `PredefinedEdgeShape`

**Type:**  `"line"|"arrow"|"tapered"|"dashed"|"dotted"`

These are valid values to ensure retro-compatibility with Ogma < 2.2

## `PropertyPath`

**Type:**  `string|Array<string>`

## `RawEdge`

**Type:**  `object`

JSON representation of an edge.

**Properties**
  * `attributes`(optional) `EdgeAttributes`
  * `data`(optional) `any`
  * `id`(optional) `EdgeId`
  * `source` `NodeId`
  * `target` `NodeId`

## `RawGraph`

**Type:**  `object`

**Properties**
  * `edges` `Array<RawEdge>`
  * `nodes` `Array<RawNode>`

## `RawNode`

**Type:**  `object`

JSON representation of a node.

**Properties**
  * `attributes`(optional) `NodeAttributes`
  * `data`(optional) `any`
  * `id`(optional) `NodeId`

## `RendererErrorCode`

**Type:**  `"NO_WEBGL"|"NO_ANGLE_INSTANCED_ARRAYS"|"OTHER"|null`

A non-null value indicates that an error has occurred and provides information on that error.
"NO_WEBGL" indicates that WebGL is not available, most likely a browser or GPU issue.
"NO_ANGLE_INSTANCED_ARRAYS" indicates that the
ANGLE_instanced_arrays
WebGL extension is not available. Also most likely a browser or GPU issue.
"OTHER" indicates an unexpected error, most likely due to a specific combination of browser/GPU/OS that was not
handled correctly by Ogma. If you happen to encounter this error code , please contact support@linkurio.us and
provide the error message along with the browser, operating system and graphics card used.

## `RendererState`

**Type:**  `"requested"|"ok"|"error"`

Indicates a renderer state.
"requested" is fired right after Ogma is initialized or the `renderer` option has been changed, and means that the
renderer has not been initialized yet.
"ok" indicates that the renderer has been initialized and runs properly.
"error" indicates that an error has occurred that prevents the renderer from running.

## `RendererType`

**Type:**  `"webgl"|"canvas"|"svg"|null`

## `SVGDrawingFunction`

**Type:**  `(elt: SVGSVGElement) => void`

The function drawing on SVG in the graph space. It is called every time the viewport is updated.

## `SVGLayer`

**Type:**  `object`

The layer object containing its properties (overloaded for handling transformed elements).

**Properties**
  * `destroy` `() => SVGLayer` Destroy the layer (remove it from the layer array, remove its listeners).
  * `element` `SVGSVGElement` SVG element used by the layer.
  * `getLevel` `() => number` Retrieves the index of the layer in the layer array.
  * `getOpacity` `() => number` Get the layer opacity, between 0 and 1.
  * `hide` `() => SVGLayer` Keep the layer in the layer array but hide its content.
  * `isHidden` `() => boolean` Check the layer visibility.
  * `moveDown` `() => SVGLayer` Take the layer down a notch (decreases its index by 1).
  * `moveTo` `(depth: number) => SVGLayer` Move the layer to the specified index in the layer array.
  * `moveToBottom` `() => SVGLayer` Move the layer at the very bottom of the array (index `0`).
  * `moveToTop` `() => SVGLayer` Move the layer at the very top of the array (index `length - 1`).
  * `moveUp` `() => SVGLayer` Take the layer up a notch (increases its index by 1).
  * `refresh` `SVGDrawingFunction` Function rerendering the SVG.
  * `setOpacity` `(opacity: number) => SVGLayer` Set the layer opacity.
  * `setPosition` `(position: {x: number, y: number}) => SVGLayer` Setter setting the element translation in the graph space.
  * `setSize` `(size: {width: number, height: number}) => SVGLayer` Setter setting the element size in the graph space.
  * `show` `() => SVGLayer` Show the layer content.

## `SVGLayerOptions`

**Type:**  `object`

SVG layer options.

**Properties**
  * `draw`(optional) `SVGDrawingFunction` Drawing function, operates in graph
coordinate space. You can render the SVG elements here or change them.
  * `element`(optional) `SVGSVGElement` SVG element to draw on. If not provided,
it will be created.

## `ScalingMethod`

**Type:**  `"scaled"|"fixed"`

## `SelectionInteractionOptions`

**Type:**  `object`

## `SimpleBoundingBox`

**Type:**  `object`

**Properties**
  * `cx` `number` X coordinate of the center of the bounding box
  * `cy` `number` Y coordinate of the center of the bounding box
  * `height` `number` Height of the bounding box
  * `maxX` `number` Maximum X coordinate of the bounding box
  * `maxY` `number` Maximum Y coordinate of the bounding box
  * `minX` `number` Minimum X coordinate of the bounding box
  * `minY` `number` Minimum Y coordinate of the bounding box
  * `width` `number` Width of the bounding box

## `Size`

**Type:**  `object`

**Properties**
  * `height` `number` Height.
  * `width` `number` Width.

## `StyleClassDefinition`

**Type:**  `object`

**Properties**
  * `options`(optional)
      * `edgeAttributes`(optional) `EdgeAttributesValue`
      * `edgeDependencies`(optional) `EdgeDependencies`
      * `edgeOutput`(optional) `EdgeOutput`
      * `nodeAttributes`(optional) `NodeAttributesValue`
      * `nodeDependencies`(optional) `NodeDependencies`
      * `nodeOutput`(optional) `NodeOutput`

## `StyleRuleDefinition`

**Type:**  `object`

**Properties**
  * `options`(optional)
      * `edgeAttributes`(optional) `EdgeAttributesValue`
      * `edgeDependencies`(optional) `EdgeDependencies`
      * `edgeOutput`(optional) `EdgeOutput`
      * `edgeSelector`(optional) `EdgeSelector`
      * `nodeAttributes`(optional) `NodeAttributesValue`
      * `nodeDependencies`(optional) `NodeDependencies`
      * `nodeOutput`(optional) `NodeOutput`
      * `nodeSelector`(optional) `NodeSelector`

## `SubGraph`

**Type:**  `object`

**Properties**
  * `object`(optional)
      * `edges`(optional) `EdgeList` Edges of the subgraph.
      * `nodes`(optional) `NodeList` Nodes of the subgraph.

## `SubLayout`

**Type:**  `object`

Represents a layout to be applied to a group of nodes. This can be:

**Properties**
  * `Custom` `{layout: string, params: object}` An object containing the layout name and its parameters.
  * `Layout` `Promise<{x: number, y: number, radius?: number}[]>` A promise that resolves to an array of points.
  * `Nothing` `null|undefined` (indicating not running any layout).

## `TextAlign`

**Type:**  `"left"|"center"`

## `TextContent`

**Type:**  `string|number|null`

## `TextPosition`

**Type:**  `'right' | 'left' | 'top' | 'bottom' | 'center'`

## `Theme`

**Type:**  `object`

**Properties**
  * `disabledEdgeAttributes`(optional) `EdgeAttributesValue` Disabled edge attributes
  * `disabledNodeAttributes`(optional) `NodeAttributesValue` Disabled node attributes
  * `edgeAttributes`(optional) `EdgeAttributes` Default edge attributes
  * `hoveredEdgeAttributes`(optional) `HoverEdgeOptions` Hovered edge attributes (you can specify duration and easing for the animation)
  * `hoveredEdgeExtremitiesAttributes`(optional) `NodeAttributesValue` Hovered edge extremities attributes
  * `hoveredNodeAttributes`(optional) `HoverNodeOptions` Hovered node attributes (you can specify duration and easing for the animation)
  * `nodeAttributes`(optional) `NodeAttributes` Default node attributes
  * `selectedEdgeAttributes`(optional) `EdgeAttributes` Selected edge attributes
  * `selectedEdgeExtremitiesAttributes`(optional) `NodeAttributesValue` Selected edge extremities attributes
  * `selectedNodeAttributes`(optional) `NodeAttributes` Selected node attributes

## `TooltipOptions`

**Type:**  `object`

**Properties**
  * `options`(optional)
      * `autoAdjust`(optional) `boolean`[default: true] When the mouse is at the edge of the screen, indicates if the
tooltip position should be corrected so it fits in the canvas.
      * `className`(optional) `string` If specified, this class name will be added to tooltip.
      * `delay`(optional) `number`[default: 0] Delay in milliseconds before the tooltip is shown when the node is hovered.
      * `position`(optional) `"top"|"bottom"|"left"|"right"|"cssDefined"`[default: "top"] Position of the tooltip relative to
the mouse. If "cssDefined" is specified, the tooltip will only be added to the graph container without
positioning it.

## `TraversalOptions`

**Type:**  `object`

**Properties**
  * `directed`(optional) `boolean`[default: false] Indicates if the graph should be considered as directed.
  * `onEdge`(optional) `(edge: Edge<ED, ND>) => void | boolean` Edge callback. Called for each edge in the traversal. If you return `false`, the edge will not be followed.
  * `onNode` `(node: Node<ND, ED>) => void | boolean` Node callback. Called for each node in the traversal. If you return `true`, the traversal is stopped, understanding that you have found what you were looking for.
  * `root` `Node<ND,ED>|NodeId` Traversal root - the node from which the traversal should start.

## `View`

**Type:**  `object`

**Properties**
  * `angle` `number`
  * `x` `number`
  * `y` `number`
  * `zoom` `number`

## `VirtualPropertiesOptions`

**Type:**  `object`

**Properties**
  * `options`(optional)
      * `edgeDataFunction`(optional) `function(edge: Edge): object` Indicates which data properties to add to a edge.
If unspecified, no data property will be added to any edge.
      * `edgeSelector`(optional) `function(edge: Edge): boolean` Indicates if the transformation should be applied
to a given edge. If unspecified, it is applied to all edges.
      * `enabled`(optional) `boolean`[default: true]
      * `nodeDataFunction`(optional) `function(node: Node): object` Indicates which data properties to add to a node.
If unspecified, no data property will be added to any node.
      * `nodeSelector`(optional) `function(node: Node): boolean` Indicates if the transformation should be applied
to a given node. If unspecified, it is applied to all nodes.

## `WatcherOptions`

**Type:**  `object|PropertyPath`

If a string or array is specified, it indicates the `path` property.

**Properties**
  * `filter`(optional) `"visible"|"all"`[default: "visible"] Indicates which elements the watcher takes into account. If "visible" (default),
only the non-filtered elements will be taken into account. If "all", all elements will be taken into account regardless of
whether they are filtered or not.
  * `path`(optional) `PropertyPath` Path of the data property to watch. If not specified, watch the root property.
  * `unwindArrays`(optional) `boolean`[default: false] If `true`, array values will be treated as multiple individual values instead of one

## `ZoomBoundaryFunction`

**Type:**  `function(params: ZoomLevelData): number`

This type of functions is passed to the Ogma options to define how to treat
the interactive zoom level depending on the node size range, graph extent
and the canvas size.

## `ZoomInteractionOptions`

**Type:**  `object`

**Properties**
  * `duration`(optional) `number`[default: 150] Indicate the duration of a manual zoom.
  * `easing`(optional) `EasingFunction`[default: "cubicOut"] Easing function to use for the zoom.
  * `enabled`(optional) `boolean`[default: true] Indicates if zoom on mouse wheel should be enabled.
  * `hideEdgeTexts`(optional) `boolean`[default: false] Indicates if the edge texts should be hidden when zooming manually.
  * `hideEdges`(optional) `boolean`[default: false] Indicates if the edges should be hidden when zooming manually.
  * `hideNodeTexts`(optional) `boolean`[default: false] Indicates if the node texts should be hidden when zooming manually.
  * `hideNodes`(optional) `boolean`[default: false] Indicates if the nodes should be hidden when zooming manually.
  * `keyboard`(optional) `boolean`[default: true] Indicates if the keyboard shortcuts for moving the view should be enabled.
  * `maxValue`(optional) `null|ZoomBoundaryFunction` Function indicating the maximum possible zoom. By default, it's not possible to zoom so the smallest node takes
more than 50% of the view. Set to `null` to remove the limit.
  * `metaKey`(optional) `'alt'|'shift'|'space'`[default: &#039;ctrl/cmd&#039;] Modifier button to use for zooming with the mousewheel or trackpad when `scrollToPan` is enabled.
  * `minValue`(optional) `null|ZoomBoundaryFunction` Function indicating the minimum possible zoom. By default, it's not possible to zoom so the graph takes less
than 20% of the view. Set to `null` to remove the limit.
  * `modifier`(optional) `number`[default: 1.8] Indicate the zoom multiplier on the manual zoom.
  * `onDoubleClick`(optional) `boolean`[default: false] Indicates if zoom on double click should be enabled.
  * `scrollToPan`(optional) `boolean`[default: false] Indicates if the view should be moved when scrolling with mousewheel or trackpad
  * `zoom`(optional) `object`

## `ZoomLevelData`

**Type:**  `object`

**Properties**
  * `biggestNodeSize` `number` Diameter of the biggest node (graph space)
  * `graphHeight` `number` Height of the graph (graph space)
  * `graphWidth` `number` Width of the graph (graph space)
  * `smallestNodeSize` `number` Diameter of the smallest node (graph space)
  * `viewWidth` `number` Width of the view (pixels)

# Events

## `addEdges`

Event triggered when some edges are added to the graph.

**Properties**
  * `edges` `EdgeList`

## `addGraph`

Event triggered when a graph is added (using addGraph method)

**Properties**
  * `edges` `EdgeList`
  * `nodes` `NodeList`

## `addNodes`

Event triggered when some nodes are added to the graph.

**Properties**
  * `nodes` `NodeList`

## `animate`

Triggered when the animation is called on nodes or edges.

**Properties**
  * `duration` `number` - The duration of the animation in milliseconds
  * `easing` `Easing`
  * `elements` `NodeList|EdgeList` - The nodes and edges affected by the animation
  * `updatesPositions` `boolean` - Whether the elements are being moved

## `beforeRemoveNodes`

Triggers right before the nodes are removed, but they are still in the
graph and their data is accessible.

**Properties**
  * `nodes` `NodeList`

## `clearGraph`

Event triggered when ogma.clearGraph is called.

**Properties**
  * `edges` `EdgeList`
  * `nodes` `NodeList`

## `click`

Event triggered when the user presses and releases a mouse button without moving in between.
Also triggers as a left button when the user presses and releases their finger (on touch devices).

**Properties**
  * `button` `MouseButton`
  * `domEvent` `Event`
  * `source` `InputSource`
  * `target` `InputTarget`
  * `x` `number`
  * `y` `number`

## `connectNodes`

Triggered when two nodes are connected using the module.

**Properties**
  * `edge` `Edge`
  * `source` `Node`
  * `target` `Node`

## `destroy`

Fired before Ogma instance is destroyed. Can be useful to remove some event listeners.

## `doubleclick`

Event triggered when the user presses and releases a mouse button twice without moving in between.

**Properties**
  * `button` `MouseButton`
  * `domEvent` `Event`
  * `source` `InputSource`
  * `target` `InputTarget`
  * `x` `number`
  * `y` `number`

## `dragEnd`

Event triggered when the user releases a mouse button, if a `onDragStart` has been emitted before.

**Properties**
  * `button` `MouseButton`
  * `domEvent` `Event`
  * `source` `InputSource`
  * `target` `InputTarget`
  * `x` `number`
  * `y` `number`

## `dragProgress`

Event triggered every time the user moves the mouse after a `onDragStart` event has been emitted,
as long as the user doesn't release the mouse.
If a node or edge was under the cursor when the first `onDragStart` event was emitted, it is passed as the `target` property.

**Properties**
  * `button` `MouseButton`
  * `domEvent` `Event`
  * `source` `InputSource`
  * `target` `InputTarget`
  * `x` `number`
  * `y` `number`

## `dragStart`

Event triggered when the user presses a mouse button and then moves the mouse (without
releasing the button).

**Properties**
  * `button` `MouseButton`
  * `domEvent` `Event`
  * `source` `InputSource`
  * `target` `InputTarget`
  * `x` `number`
  * `y` `number`

## `drop`

Triggered when the user drops an element into the Ogma container. Note that x and y arguments are Graph coordinates.

**Properties**
  * `domEvent` `DragEvent`
  * `x` `number`
  * `y` `number`

## `edgeClassAdded_MyClass`

Triggered when the specified class (MyClass) is added to some edges.

**Properties**
  * `edges` `EdgeList`

## `edgeClassRemoved_MyClass`

Triggered when the specified class (MyClass) is removed from some edges.

**Properties**
  * `edges` `EdgeList`

## `edgesDisabled`

Event triggered when edges are disabled

**Properties**
  * `edges` `EdgeList`

## `edgesEnabled`

Event triggered when edges are enabled

**Properties**
  * `edges` `EdgeList`

## `edgesSelected`

Event triggered when edges are selected

**Properties**
  * `edges` `EdgeList`

## `edgesUnSelected`

Event triggered when edges are unselected

**Properties**
  * `edges` `EdgeList`

## `frame`

Event triggered when a frame is rendered.

## `geoDisabled`

Triggered when the geo mode is switched off

## `geoEnabled`

Triggered when the geo mode is activated

## `geoLoaded`

Triggered when the background map images are loaded

## `gestureEnd`

Event triggered when the user stop touching the screen with two fingers.

**Properties**
  * `domEvent` `Event`

## `gestureProgress`

Event triggered when the users moves two fingers.

**Properties**
  * `angle` `number`
  * `domEvent` `Event`
  * `dx` `number`
  * `dy` `number`
  * `scale` `number`
  * `x` `number`
  * `y` `number`

## `gestureStart`

Event triggered when the user touch the screen with two fingers.

**Properties**
  * `domEvent` `Event`

## `idle`

Triggered when the graph is idle (no more camera movements, no more layout iterations).

## `keydown`

Event triggered when the user presses the keyboard button.

**Properties**
  * `domEvent` `KeyboardEvent`
  * `key` `string`
  * `keyCode` `number`

## `keyup`

Event triggered when the user releases the keyboard button.

**Properties**
  * `domEvent` `KeyboardEvent`
  * `key` `string`
  * `keyCode` `number`

## `layoutComputed`

This event is fired after the layout algorithm has finished the calculations,
but before the positions are applied. Use it for UI interactions, because if
you would add position manipulations into the listener, they can interfere
with the layout results.

**Properties**
  * `name` `string`

## `layoutEnd`

Trigger the specified function when two nodes are connected using the module.

**Properties**
  * `ids` `Array<NodeId>`
  * `name` `string`
  * `positions` `{ before: Point[], after: Point[]}`

## `layoutStart`

Triggered when two nodes are connected using the module.

**Properties**
  * `ids` `Array<NodeId>`
  * `name` `string`

## `mousedown`

Event triggered when the user presses a mouse button. Also triggers as a left button when the user
presses their finger (on touch devices).

**Properties**
  * `button` `MouseButton`
  * `domEvent` `Event`
  * `source` `InputSource`
  * `x` `number`
  * `y` `number`

## `mousemove`

Event triggered when the user moves the mouse (or their finger in touch devices).

**Properties**
  * `domEvent` `Event`
  * `dx` `number`
  * `dy` `number`
  * `source` `InputSource`
  * `x` `number`
  * `y` `number`

## `mouseout`

Event triggered when a node or edge stops being hovered.

**Properties**
  * `domEvent` `Event` - The original DOM event.
  * `source` `InputSource` - The source of the event.
  * `target` `InputTarget` - The target of the event.
  * `x` `number` - The x coordinate of the mouse.
  * `y` `number` - The y coordinate of the mouse.

## `mouseover`

Event triggered when a node or edge is hovered.

**Properties**
  * `domEvent` `Event`
  * `source` `InputSource`
  * `target` `InputTarget`
  * `x` `number`
  * `y` `number`

## `mouseup`

Event triggered when the user releases a mouse button. Also triggers as a left button when the user
releases their finger (on touch devices).

**Properties**
  * `button` `MouseButton`
  * `domEvent` `Event`
  * `source` `InputSource`
  * `target` `InputTarget`
  * `x` `number`
  * `y` `number`

## `mousewheel`

Event triggered when the user uses the mouse wheel.

**Properties**
  * `delta` `number` - The delta of the mouse wheel.
  * `domEvent` `Event` - The original DOM event.
  * `source` `InputSource` - The source of the event.
  * `target` `InputTarget` - The target of the event.
  * `x` `number` - The x coordinate of the mouse.
  * `y` `number` - The y coordinate of the mouse.

## `move`

Event triggered when camera is moving.

## `nodeClassAdded_MyClass`

Triggered when the specified class (MyClass) is added to some nodes.

**Properties**
  * `nodes` `NodeList`

## `nodeClassRemoved_MyClass`

Triggered when the specified class (MyClass) is removed from some nodes.

**Properties**
  * `nodes` `NodeList`

## `nodesDisabled`

Event triggered when nodes are disabled

**Properties**
  * `nodes` `NodeList`

## `nodesDragEnd`

Triggered when the user stop dragging some nodes.

**Properties**
  * `end` `Array<{x: number, y: number}>`
  * `nodes` `NodeList`
  * `start` `Array<{x: number, y: number}>`

## `nodesDragProgress`

Triggered when the user drags some nodes.

**Properties**
  * `dx` `number`
  * `dy` `number`
  * `nodes` `NodeList`

## `nodesDragStart`

Triggered when the user starts to drag some nodes.

**Properties**
  * `nodes` `NodeList`

## `nodesEnabled`

Event triggered when nodes are enabled

**Properties**
  * `nodes` `NodeList`

## `nodesSelected`

Event triggered when nodes are selected

**Properties**
  * `nodes` `NodeList`

## `nodesUnselected`

Event triggered when nodes are unselected

**Properties**
  * `nodes` `NodeList`

## `refreshStyleRule`

Property {StyleRule} rule Rule that has been applied

## `removeEdges`

Event triggered when some edges are removed from the graph.

**Properties**
  * `edges` `EdgeList`

## `removeNodes`

Event triggered when some nodes are removed from the graph.

**Properties**
  * `nodes` `NodeList`

## `rendererStateChange`

Triggered when the renderer is requested, successfully initialized or encounters an error.

**Properties**
  * `code` `RendererErrorCode`
  * `message` `string`
  * `state` `RendererState`
  * `type` `RendererType`

## `resize`

**Properties**
  * `difHeigh` `number`
  * `difWidth` `number`
  * `height,` `number`
  * `prevHeight` `number`
  * `prevWidth` `number`
  * `width` `number`

## `rotate`

Event triggered when camera is moving.

## `tooltipHide`

Event triggered when a tooltip is hidden.

**Properties**
  * `tooltip` `HTMLElement`

## `tooltipShow`

Event triggered when a tooltip is shown.

**Properties**
  * `tooltip` `HTMLElement`

## `transformationDestroyed`

Triggered when a transformation is destroyed

**Properties**
  * `target` `Transformation`

## `transformationDisabled`

Triggered when a transformation is disabled

**Properties**
  * `target` `Transformation`

## `transformationEnabled`

Triggered when a transformation is activated

**Properties**
  * `target` `Transformation`

## `transformationRefresh`

Triggered when a transformation is refreshed

**Properties**
  * `target` `Transformation`

## `transformationSetIndex`

Triggered when a transformation index is set

**Properties**
  * `target` `Transformation`

## `updateEdgeData`

**Properties**
  * `changes` `changes: Array<{property: PropertyPath, edges: EdgeList, previousValues: Array<any>, newValues: Array<any>}>`

## `updateNodeData`

Trigger the specified function when the data of some nodes is updated.

**Properties**
  * `changes` `Array<{property: PropertyPath, nodes: NodeList, previousValues: Array<any>, newValues: Array<any>}>`

## `viewChanged`

Event triggered when a camera movement (zoom, panning, rotation) is finished.

## `viewExportStart`

Event triggered when exporting the graph in png, jpg or svg.

**Properties**
  * `camera` `Camera`

## `zoomStart`

Triggered when camera starts zoom.

**Properties**
  *
  * `duration` `number`
  * `easing` `Easing`
  * `endZoom` `number` Target zoom leve
  * `startZoom` `number` Zoom level at the start of the zoom.

